<?php

declare(strict_types=1);

namespace Drupal\oauth_client\Form;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\oauth_client\Access\AccessTrait;
use Drupal\oauth_client\Entity\NotificationType;
use Drupal\oauth_client\Entity\OauthClientRequestInterface;
use Drupal\oauth_client\Entity\OauthClientRequestStatus;
use Drupal\oauth_client\Entity\OauthClientRequestTypeInterface;
use Drupal\oauth_client\Event\NotificationEvent;
use Drupal\user\UserInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Form controller for the OAuth Client Request entity edit forms.
 */
class OauthClientRequestForm extends ContentEntityForm {

  use AccessTrait;
  use AutowireTrait;

  /**
   * The target user from the route parameter.
   */
  protected ?UserInterface $userFromRoute = NULL;

  public function __construct(
    EntityRepositoryInterface $entity_repository,
    EntityTypeBundleInfoInterface $entity_type_bundle_info,
    TimeInterface $time,
    protected EventDispatcherInterface $eventDispatcher,
  ) {
    parent::__construct($entity_repository, $entity_type_bundle_info, $time);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('entity.repository'),
      $container->get('entity_type.bundle.info'),
      $container->get('datetime.time'),
      $container->get('event_dispatcher'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?UserInterface $user = NULL) {
    $this->userFromRoute = $user;
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    $form = parent::form($form, $form_state);

    /**
     * @var \Drupal\oauth_client\Entity\OauthClientRequestInterface $entity
     */
    $entity = $this->entity;
    $isManager = $this->isManager($this->currentUser());
    $isNew = $entity->isNew();

    // Add request type information at the top of the form.
    $this->addRequestTypeInformation($form, $entity);

    // Set user field from route parameter.
    $this->setUserFromRoute($form);

    // Configure form based on user type and entity state.
    match (TRUE) {
      $isManager && $isNew => $this->configureManagerAddForm($form),
      $isManager && !$isNew => $this->configureManagerEditForm($form, $entity),
      default => $this->configureUserAddForm($form, $isNew),
    };

    return $form;
  }

  /**
   * Add a request type label and description to the form.
   */
  protected function addRequestTypeInformation(array &$form, OauthClientRequestInterface $entity): void {
    $bundleId = $entity->bundle();
    $requestType = $this->entityTypeManager
      ->getStorage('oauth_client_request_type')
      ->load($bundleId);

    if ($requestType instanceof OauthClientRequestTypeInterface) {
      $form['request_type_info'] = [
        '#type' => 'container',
        '#weight' => -100,
        '#attributes' => ['class' => ['oauth-client-request-type-info']],
      ];

      $form['request_type_info']['label'] = [
        '#type' => 'html_tag',
        '#tag' => 'h3',
        '#value' => $requestType->label(),
      ];

      if ($description = $requestType->getDescription()) {
        $form['request_type_info']['description'] = [
          '#type' => 'html_tag',
          '#tag' => 'p',
          '#value' => $description,
        ];
      }
    }
  }

  /**
   * Sets the user field from route parameter.
   */
  protected function setUserFromRoute(array &$form): void {
    if ($this->userFromRoute) {
      $form['user']['widget'][0]['target_id']['#default_value'] = $this->userFromRoute;
    }
  }

  /**
   * Configures the form for managers adding a new request.
   */
  protected function configureManagerAddForm(array &$form): void {
    $this->hideFields($form, ['user', 'request_reason', 'reject_reason', 'status']);
    $form['status']['widget']['#default_value'] = OauthClientRequestStatus::Active->value;
  }

  /**
   * Configures the form for managers editing existing requests.
   */
  protected function configureManagerEditForm(array &$form, OauthClientRequestInterface $entity): void {
    $this->disableFields($form, ['user', 'label', 'request_reason', 'user']);
    $this->configureStatusField($form, $entity);
    $this->configureRejectReasonField($form);
  }

  /**
   * Configures status field options based on current status.
   */
  protected function configureStatusField(array &$form, OauthClientRequestInterface $entity): void {
    $currentStatus = $entity->getStatus();
    $statusOptions = $currentStatus === OauthClientRequestStatus::Rejected
      ? [OauthClientRequestStatus::Active->value => t('Active')]
      : [
        OauthClientRequestStatus::Active->value => t('Active'),
        OauthClientRequestStatus::Rejected->value => t('Rejected'),
      ];

    $form['status']['widget']['#title'] = $this->t('Current status: %status', [
      '%status' => OauthClientRequestStatus::getOptions()[$form['status']['widget']['#default_value']],
    ]);
    $form['status']['widget']['#options'] = $statusOptions;
    $form['status']['widget']['#required'] = TRUE;
  }

  /**
   * Configures reject reason field with conditional states.
   */
  protected function configureRejectReasonField(array &$form): void {
    $form['reject_reason']['#states'] = [
      'visible' => [':input[name="status"]' => ['value' => OauthClientRequestStatus::Rejected->value]],
      'required' => [':input[name="status"]' => ['value' => OauthClientRequestStatus::Rejected->value]],
    ];
  }

  /**
   * Configures the form for regular users adding a new request.
   */
  protected function configureUserAddForm(array &$form, bool $isNew): void {
    $this->hideFields($form, ['user', 'reject_reason', 'status']);
    if ($isNew) {
      $form['status']['widget']['#default_value'] = OauthClientRequestStatus::Pending->value;
    }
    $form['request_reason']['widget'][0]['value']['#required'] = TRUE;
  }

  /**
   * Hides specified fields in the form.
   */
  protected function hideFields(array &$form, array $fieldNames): void {
    foreach ($fieldNames as $fieldName) {
      $form[$fieldName]['#access'] = FALSE;
    }
  }

  /**
   * Disables specified fields in the form.
   */
  protected function disableFields(array &$form, array $fieldNames): void {
    foreach ($fieldNames as $fieldName) {
      $form[$fieldName]['#disabled'] = TRUE;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    if (!$this->entity->isNew()) {
      $this->validateStatusTransition($form, $form_state);
    }
  }

  /**
   * Validates status transition and reject reason.
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  protected function validateStatusTransition(array $form, FormStateInterface $form_state): void {
    $newStatus = $form_state->getValue('status')[0]['value'] ?? '';

    if ($newStatus === OauthClientRequestStatus::Rejected->value) {
      $rejectReason = trim($form_state->getValue('reject_reason')[0]['value'] ?? '');
      if (empty($rejectReason)) {
        $form_state->setErrorByName('reject_reason', $this->t('Reject reason is required when rejecting a request.'));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $entity = $this->entity;
    $originalStatus = $entity->isNew() ? NULL : $entity->getStatus();

    parent::submitForm($form, $form_state);

    $newStatus = $this->getEntity()->getStatus();
    $notificationAction = $this->calculateNotificationAction($newStatus, $originalStatus);
    // Store the action temporarily to be able to dispatch it after the entity
    // is saved.
    $form_state->setTemporaryValue('notification_action', $notificationAction);
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state): int {
    /** @var \Drupal\oauth_client\Entity\OauthClientRequestInterface $entity */
    $entity = $this->getEntity();
    $result = parent::save($form, $form_state);

    $label = $entity->label();

    $this->addStatusMessage($result, $label);
    $this->logSaveAction($result, $label);
    $form_state->setRedirect('view.oauth_client_request.user_requests', [
      'user' => $this->userFromRoute->id(),
    ]);

    $notificationAction = $form_state->getTemporaryValue('notification_action');
    if ($notificationAction) {
      $this->eventDispatcher->dispatch(new NotificationEvent($entity, $notificationAction));
    }

    return $result;
  }

  /**
   * Calculates the action for the emails.
   *
   * @param \Drupal\oauth_client\Entity\OauthClientRequestStatus $newStatus
   *   The new status of the entity.
   * @param \Drupal\oauth_client\Entity\OauthClientRequestStatus|null $originalStatus
   *   The original status of the entity.
   *
   * @return \Drupal\oauth_client\Entity\NotificationType|null
   *   The action for the email.
   */
  protected function calculateNotificationAction(OauthClientRequestStatus $newStatus, ?OauthClientRequestStatus $originalStatus = NULL): ?NotificationType {
    if ($originalStatus === NULL) {
      return NotificationType::Request;
    }

    $statuses = [$newStatus, $originalStatus];
    return match ($statuses) {
      [OauthClientRequestStatus::Active, OauthClientRequestStatus::Pending] => NotificationType::Approve,
      [OauthClientRequestStatus::Rejected, OauthClientRequestStatus::Pending] => NotificationType::Reject,
      default => NULL,
    };
  }

  /**
   * Adds appropriate status message based on save result.
   */
  protected function addStatusMessage(int $result, string $label): void {
    $message = match ($result) {
      SAVED_NEW => $this->t('New OAuth client request %label has been created.', ['%label' => $label]),
      SAVED_UPDATED => $this->t('The OAuth client request %label has been updated.', ['%label' => $label]),
      default => NULL,
    };

    if ($message) {
      $this->messenger()->addStatus($message);
    }
  }

  /**
   * Logs the save action.
   */
  protected function logSaveAction(int $result, string $label): void {
    $message = match ($result) {
      SAVED_NEW => 'Created new OAuth client request %label.',
      SAVED_UPDATED => 'Updated OAuth client request %label.',
      default => NULL,
    };

    if ($message) {
      $this->logger('oauth_client')->notice($message, ['%label' => $label]);
    }
  }

  /**
   * Gets the title for the form page.
   *
   * @param \Drupal\user\UserInterface $user
   *   The targeted user.
   * @param \Drupal\Core\Routing\RouteMatchInterface $routeMatch
   *   The route match.
   * @param \Drupal\oauth_client\Entity\OauthClientRequestTypeInterface|null $oauth_client_request_type
   *   The bundle on new entity creation.
   *
   * @return string|\Stringable
   *   The page title.
   */
  public function title(UserInterface $user, RouteMatchInterface $routeMatch, ?OauthClientRequestTypeInterface $oauth_client_request_type = NULL): string|\Stringable {
    $entity = $routeMatch->getParameter('oauth_client_request');

    if ($entity instanceof OauthClientRequestInterface) {
      return $this->t('Edit the client request %label', ['%label' => $entity->label()]);
    }

    $args = [
      '%type' => $oauth_client_request_type->label(),
      '%user' => $user->getDisplayName(),
    ];

    if ($this->isManager($this->currentUser())) {
      if ($user->id() != $this->currentUser()->id()) {
        return $this->t('Create a new %type client for %user', $args);
      }
      return $this->t('Create a new %type client', $args);
    }

    return $this->t('Request a new %type client', [
      '%type' => $oauth_client_request_type->label(),
    ]);
  }

}
