<?php

declare(strict_types=1);

namespace Drupal\oauth_client\Form;

use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Entity\BundleEntityFormBase;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\oauth_client\Entity\GrantType;
use Drupal\oauth_client\Entity\OauthClientRequestType;
use Drupal\oauth_client\Entity\OauthClientRequestTypeInterface;
use Drupal\simple_oauth\Oauth2ScopeInterface;
use Drupal\simple_oauth\Oauth2ScopeProviderInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * OAuth2 Client Request Type entity form.
 */
class OauthClientRequestTypeForm extends BundleEntityFormBase {

  use AutowireTrait;

  public function __construct(
    #[Autowire(service: 'simple_oauth.oauth2_scope.provider')]
    protected Oauth2ScopeProviderInterface $oauth2ScopeProvider,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    $form = parent::form($form, $form_state);

    $form['label'] = [
      '#title' => $this->t('Label'),
      '#type' => 'textfield',
      '#default_value' => $this->getEntity()->label(),
      '#required' => TRUE,
      '#size' => 30,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $this->getEntity()->id(),
      '#maxlength' => EntityTypeInterface::BUNDLE_MAX_LENGTH,
      '#machine_name' => [
        'exists' => [OauthClientRequestType::class, 'load'],
        'source' => ['label'],
      ],
      '#description' => $this->t('Unique internal ID: lowercase letters, numbers, and underscores only.'),
      '#required' => TRUE,
    ];

    $form['description'] = [
      '#title' => $this->t('Description'),
      '#type' => 'textarea',
      '#default_value' => $this->getEntity()->getDescription(),
    ];

    $form['grant_type'] = [
      '#type' => 'details',
      '#tree' => TRUE,
      '#title' => $this->t('Grant types'),
      '#description' => $this->t("NOTE: The 'Authorization Code' grant type is not supported yet because it requires <a href=\"@issue\">this change</a> to be adopted by the <a href=\"@module\">Simple OAuth (OAuth2) & OpenID Connect</a> module.", [
        '@issue' => 'https://www.drupal.org/i/3398468',
        '@module' => 'https://www.drupal.org/project/simple_oauth',
      ]),
      '#required' => TRUE,
      '#open' => TRUE,
    ];

    foreach (GrantType::cases() as $grantType) {
      // Temporarily disable authorization_code grant type.
      // See https://www.drupal.org/i/3398468.
      if ($grantType === GrantType::AuthorizationCode) {
        continue;
      }

      $form['grant_type'][$grantType->value]['enabled'] = [
        '#type' => 'checkbox',
        '#title' => $grantType->label(),
        '#default_value' => $this->getEntity()->getGrantTypes()[$grantType->value]['enabled'],
      ];

      if ($grantTypeSettings = $this->getGrantTypeSettingsForm($grantType)) {
        $form['grant_type'][$grantType->value]['settings'] = [
          '#type' => 'fieldset',
          '#tree' => TRUE,
          '#title' => $this->t('Settings'),
          '#title_display' => 'invisible',
          '#states' => [
            'visible' => [
              ':input[name="grant_type[' . $grantType->value . '][enabled]"]' => ['checked' => TRUE],
            ],
          ],
          '#open' => TRUE,
        ] + $grantTypeSettings;
      }
      else {
        $form['grant_type'][$grantType->value]['settings'] = [
          '#type' => 'value',
          '#value' => [],
        ];
      }
    }

    $form['scope'] = [
      '#type' => 'select',
      '#title' => $this->t('Scope'),
      '#options' => $this->getScopes(),
      '#default_value' => $this->getEntity()->getScopeId(),
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * Returns the grant type subform.
   */
  protected function getGrantTypeSettingsForm(GrantType $grantType): array {
    if ($grantType === GrantType::AuthorizationCode) {
      return [
        'refresh_token' => [
          '#type' => 'checkbox',
          '#title' => $this->t('Refresh token'),
          '#default_value' => $this->getEntity()->getGrantTypeSetting($grantType, 'refresh_token'),
        ],
        'automatic_authorization' => [
          '#type' => 'checkbox',
          '#title' => $this->t('Automatic authorization'),
          '#default_value' => $this->getEntity()->getGrantTypeSetting($grantType, 'automatic_authorization'),
        ],
        'remember_approval' => [
          '#type' => 'checkbox',
          '#title' => $this->t('Remember approval'),
          '#default_value' => $this->getEntity()->getGrantTypeSetting($grantType, 'remember_approval'),
        ],
        'pkce' => [
          '#type' => 'checkbox',
          '#title' => $this->t('Use PKCE?l'),
          '#default_value' => $this->getEntity()->getGrantTypeSetting($grantType, 'pkce'),
        ],
      ];
    }
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    if (
      !$form_state->getValue(['grant_type', GrantType::ClientCredentials->value, 'enabled']) &&
      !$form_state->getValue(['grant_type', GrantType::AuthorizationCode->value, 'enabled'])
    ) {
      $form_state->setErrorByName('grant_type', $this->t('At least one grant type must be enabled.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    parent::submitForm($form, $form_state);
    $form_state->setRedirect('entity.oauth_client_request_type.collection');
    $this->messenger()->addStatus($this->t('The OAuth2 client request type %label has been updated.', [
      '%label' => $this->getEntity()->label(),
    ]));
  }

  /**
   * {@inheritdoc}
   */
  public function getEntity(): OauthClientRequestTypeInterface {
    $entity = parent::getEntity();

    // Extend this class only for the sake of strict typing return.
    assert($entity instanceof OauthClientRequestTypeInterface);
    return $entity;
  }

  /**
   * Returns a list of scopes to be used as select options.
   *
   * @return array<array-key, string|\Stringable>
   *   Associative list of scope descriptions keyed by scope IDs.
   */
  protected function getScopes(): array {
    return array_map(
      fn(Oauth2ScopeInterface $scope): string => $scope->getDescription(),
      $this->oauth2ScopeProvider->loadMultiple(),
    );
  }

}
