<?php

declare(strict_types=1);

namespace Drupal\oauth_client\Hook;

use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Render\BubbleableMetadata;
use Drupal\Core\Routing\UrlGeneratorInterface;
use Drupal\Core\Url;
use Drupal\user\Entity\User;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Define oauth_client related hooks.
 */
class OauthClientTokenHooks {

  public function __construct(
    protected DateFormatterInterface $dateFormatter,
    #[Autowire(service: 'url_generator')]
    protected UrlGeneratorInterface $urlGenerator,
  ) {}

  /**
   * Provides tokens for the oauth_client module.
   *
   * @phpstan-ignore-next-line as Attribute class
   *   Drupal\Core\Hook\Attribute\Hook does not exist.
   */
  #[Hook('token_info')]
  public function tokenInfo(): array {
    return [
      'types' => [
        'oauth_client' => [
          'name' => t('OAuth Client'),
          'description' => t('Tokens related to OAuth Client module.'),
        ],
        'oauth_client_request' => [
          'name' => t('OAuth Client Request'),
          'description' => t('Tokens related to OAuth Client Request entity.'),
        ],
      ],
      'tokens' => [
        'oauth_client' => [
          'client_requests' => [
            'name' => t('User requests (admin)'),
            'description' => t('Link to the user requests admin page.'),
          ],
        ],
        'oauth_client_request' => [
          'label' => [
            'name' => t('Label'),
            'description' => t('The label of the request.'),
          ],
          'request-edit-url' => [
            'name' => t('Request edit URL'),
            'description' => t('The URL to edit the request. Usually for the manager to approve or reject the request.'),
          ],
          'client-edit-url' => [
            'name' => t('Client edit URL'),
            'description' => t('The URL the client to change the the client of the request. Usually for the owner to edit the secret.'),
          ],
          'client' => [
            'name' => t('Client'),
            'description' => t('The client the request is applying for.'),
          ],
          'owner' => [
            'name' => t('Owner'),
            'description' => t('The user who made the request.'),
          ],
          'owner:client_requests' => [
            'name' => t('Owner requests'),
            'description' => t('Link to the user requests page.'),
          ],
          'request_reason' => [
            'name' => t('Request reason'),
            'description' => t('The reason why the request was made.'),
          ],
          'reject_reason' => [
            'name' => t('Reject reason'),
            'description' => t('The reason why the request was rejected.'),
          ],
          'created' => [
            'name' => t('Created date'),
            'description' => t('The date the request was created.'),
            'type' => 'date',
          ],
          'changed' => [
            'name' => t('Changed date'),
            'description' => t('The date the request was last changed.'),
            'type' => 'date',
          ],
        ],
      ],
    ];
  }

  /**
   * Implements hook_tokens().
   *
   * @phpstan-ignore-next-line as Attribute class
   *   Drupal\Core\Hook\Attribute\Hook does not exist.
   */
  #[Hook('tokens')]
  public function tokens(string $type, array $tokens, array $data, array $options, BubbleableMetadata $bubbleableMetadata): array {
    $replacements = [];
    $urlOptions = ['absolute' => TRUE];

    if ($type === 'oauth_client') {
      foreach ($tokens as $name => $original) {
        switch ($name) {
          case 'client_requests':
            $replacements[$original] = $this->urlGenerator->generateFromRoute('view.oauth_client_request.admin_page', [], $urlOptions);
            break;

          default:
            break;
        }
      }
    }
    elseif ($type === 'oauth_client_request' && !empty($data['oauth_client_request'])) {
      /** @var \Drupal\oauth_client\Entity\OauthClientRequestInterface $request */
      $request = $data['oauth_client_request'];
      foreach ($tokens as $name => $original) {
        switch ($name) {
          case 'label':
            $replacements[$original] = $request->label();
            break;

          case 'request-edit-url':
            $replacements[$original] = $request->toUrl('edit-form')->setAbsolute()->toString();
            break;

          case 'client-edit-url':
            $url = Url::fromRoute('oauth_client.edit', [
              'user' => $request->getUser()->id(),
              'client' => $request->getClient()->id(),
            ]);
            $replacements[$original] = $url->setAbsolute()->toString();
            break;

          case 'request_reason':
            $replacements[$original] = $request->getRequestReason();
            break;

          case 'reject_reason':
            $replacements[$original] = $request->getRejectReason();
            break;

          case 'client':
            $client = $request->getClient();
            $bubbleableMetadata->addCacheableDependency($client);
            $replacements[$original] = $client->label();
            break;

          case 'owner':
            $account = $request->getOwner() ?: User::load(0);
            $bubbleableMetadata->addCacheableDependency($account);
            $replacements[$original] = $account->label();
            break;

          case 'owner:client_requests':
            $account = $request->getOwner() ?: User::load(0);
            $bubbleableMetadata->addCacheableDependency($account);
            $replacements[$original] = $this->urlGenerator->generateFromRoute('view.oauth_client_request.user_requests', ['user' => $account->id()], $urlOptions);
            break;

          case 'created':
            $replacements[$original] = $this->dateFormatter->format($request->getCreatedTime());
            break;

          case 'changed':
            $replacements[$original] = $this->dateFormatter->format($request->getChangedTime());
            break;

          default:
            break;
        }
      }
    }

    return $replacements;
  }

}
