<?php

declare(strict_types=1);

namespace Drupal\oauth_client\Permission;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\oauth_client\Entity\OauthClientRequestTypeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Permissions callback.
 */
class OauthClientPermissions implements ContainerInjectionInterface {

  use StringTranslationTrait;

  protected function __construct(
    protected readonly EntityTypeManagerInterface $entityTypeManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static($container->get('entity_type.manager'));
  }

  /**
   * {@inheritdoc}
   */
  public function __invoke(): array {
    $permissions = [];

    foreach ($this->entityTypeManager->getStorage('oauth_client_request_type')->loadMultiple() as $type) {
      $dependencies[$type->getConfigDependencyKey()] = [$type->getConfigDependencyName()];

      $permissions[static::getUserPermission($type)] = [
        'title' => $this->t('Apply for %label oauth2 client', [
          '%label' => $type->label(),
        ]),
        'dependencies' => [
          $type->getConfigDependencyKey() => [$type->getConfigDependencyName()],
        ],
      ];
    }

    return $permissions;
  }

  /**
   * Returns the permission string for requesting a client with a given type.
   *
   * @param \Drupal\oauth_client\Entity\OauthClientRequestTypeInterface|string $type
   *   The type as entity or entity ID.
   */
  public static function getUserPermission(OauthClientRequestTypeInterface|string $type): string {
    $id = $type instanceof OauthClientRequestTypeInterface ? $type->id() : $type;
    return "request $id oauth2 client";
  }

}
