<?php

declare(strict_types=1);

namespace Drupal\oauth_client\Controller;

use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Drupal\user\UserInterface;

/**
 * Controller for oauth_client.user route.
 */
class OauthClientUserController extends ControllerBase {

  /**
   * Provides a page callback for oauth_client.user route.
   *
   * @param \Drupal\user\UserInterface $user
   *   The user account.
   *
   * @return array
   *   The page render array.
   */
  public function __invoke(UserInterface $user): array {
    $isCurrentUser = $user->id() == $this->currentUser()->id();
    $args = ['@name' => $user->getDisplayName()];

    $storage = $this->entityTypeManager()->getStorage('consumer');
    $clientIds = $storage->getQuery()
      ->accessCheck()
      ->condition('user_id', $user->id())
      ->condition('confidential', TRUE)
      ->sort('label')
      ->execute();

    $clients = [];
    foreach ($storage->loadMultiple($clientIds) as $client) {
      $clients[] = [
        'id' => $client->getClientId(),
        'label' => $client->label(),
        'operations' => [
          'data' => [
            '#type' => 'operations',
            '#links' => [
              'edit' => [
                'title' => $this->t('Edit'),
                'url' => Url::fromRoute('oauth_client.edit', [
                  'user' => $user->id(),
                  'client' => $client->id(),
                ]),
              ],
              'delete' => [
                'title' => t('Delete'),
                'url' => Url::fromRoute('oauth_client.delete', [
                  'user' => $user->id(),
                  'client' => $client->id(),
                ]),
              ],
            ],
          ],
        ],
      ];
    }

    $build = [
      [
        '#markup' => $this->t('Clients'),
        '#prefix' => '<h2>',
        '#suffix' => '</h2>',
      ],
      [
        '#theme' => 'table',
        '#header' => [
          'client_id' => $this->t('Client ID'),
          'label' => $this->t('Name'),
          'operations' => $this->t('Operations'),
        ],
        '#rows' => $clients,
        '#empty' => $isCurrentUser ?
        $this->t('No client apps are assigned to you.') :
        $this->t('No client apps are assigned to @name.', $args),
      ],
    ];

    (new CacheableMetadata())
      ->addCacheContexts(['route'])
      ->applyTo($build);

    return $build;
  }

  /**
   * Provides an access callback for oauth_client.user route.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result object.
   */
  public function access(UserInterface $user): AccessResultInterface {
    // Rely on the 'add-page' route access.
    $url = Url::fromRoute('entity.oauth_client_request.add_page', ['user' => $user->id()]);
    return $url->access(return_as_object: TRUE);
  }

}
