<?php

declare(strict_types=1);

namespace Drupal\oauth_client\Form;

use Drupal\consumers\Entity\ConsumerInterface;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\user\UserInterface;

/**
 * Form for editing an OAuth2 client.
 */
class OauthClientEditForm extends FormBase {

  use FormAccessTrait;

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?UserInterface $user = NULL, ?ConsumerInterface $client = NULL): array {
    $form['user'] = [
      '#type' => 'value',
      '#value' => $user,
    ];

    $form['client'] = [
      '#type' => 'value',
      '#value' => $client,
    ];

    $form['client_id'] = [
      '#type' => 'item',
      '#title' => $this->t('Client ID'),
      '#markup' => $client->getClientId(),
    ];

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#required' => TRUE,
      // If label is temporary, enforce a custom label.
      '#default_value' => str_starts_with($client->label(), '[temporary name] ') ? '' : $client->label(),
    ];

    $form['secret'] = [
      '#type' => 'password',
      '#title' => t('New secret'),
      '#description' => $this->t('Use this field to create a hash of the secret key. This module will never store your client secret key, only a hash of it. Leave it empty to preserve the current secret.'),
    ];

    $form['actions'] = [
      '#type' => 'actions',
      'submit' => [
        '#type' => 'submit',
        '#value' => $this->t('Save'),
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $user = $form_state->getValue('user');
    $client = $form_state->getValue('client');

    $client->set('label', $form_state->getValue('label'));

    // If nothing has been entered, ignore the change.
    if ($secret = $form_state->getValue('secret')) {
      $client->set('secret', $secret);
    }

    $client->save();

    $this->messenger()
      ->addStatus($this->t('The %client OAuth2 client has been updated.', ['%client' => $client->label()]));

    $form_state->setRedirect('oauth_client.user', ['user' => $user->id()]);
  }

  /**
   * Checks access to the form.
   *
   * @param \Drupal\user\UserInterface $user
   *   The user account.
   * @param \Drupal\consumers\Entity\ConsumerInterface $client
   *   The OAuth2 client.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result object.
   */
  public function access(UserInterface $user, ConsumerInterface $client): AccessResultInterface {
    return $this->checkAccess('edit', $user, $client);
  }

  /**
   * {@inheritdoc}
   */
  public function title(ConsumerInterface $client): TranslatableMarkup {
    return $this->t('%label OAuth2 client', ['%label' => $client->label()]);
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'oauth_client_edit';
  }

}
