<?php

declare(strict_types=1);

namespace Drupal\oauth_client\Hook;

use Drupal\consumers\Entity\ConsumerInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\oauth_client\Entity\OauthClientRequestInterface;
use Drupal\user\UserInterface;

/**
 * Entity deletion handler for OAuth Client module.
 */
readonly class OauthClientEntityDeleteHandler {

  /**
   * Constructs a new OauthClientEntityDeleteHandler.
   */
  public function __construct(
    private EntityTypeManagerInterface $entityTypeManager,
  ) {}

  /**
   * Implements hook_entity_delete().
   *
   * @phpstan-ignore-next-line as Attribute class Drupal\Core\Hook\Attribute\Hook does not exist.
   */
  #[Hook('entity_delete')]
  public function entityDelete(EntityInterface $entity): void {
    match (TRUE) {
      $entity instanceof OauthClientRequestInterface => $this->deleteLinkedConsumer($entity),
      $entity instanceof ConsumerInterface => $this->deleteLinkedRequests($entity),
      $entity instanceof UserInterface => $this->handleUserAccountDelete($entity),
      default => NULL,
    };
  }

  /**
   * Deletes the consumer linked to an oauth client request.
   */
  protected function deleteLinkedConsumer(OauthClientRequestInterface $entity): void {
    $consumer = $entity->getClient();
    if (!$consumer instanceof ConsumerInterface) {
      return;
    }
    $consumer->delete();
  }

  /**
   * Deletes oauth client requests linked to a consumer.
   */
  protected function deleteLinkedRequests(ConsumerInterface $entity): void {
    $storage = $this->entityTypeManager->getStorage('oauth_client_request');
    $query = $storage->getQuery()
      ->condition('client', $entity->id())
      ->accessCheck(FALSE);
    $requestIds = $query->execute();

    if (empty($requestIds)) {
      return;
    }
    $requests = $storage->loadMultiple($requestIds);
    $storage->delete($requests);
  }

  /**
   * Handles deletion of user accounts.
   *
   * @param \Drupal\user\UserInterface $user
   *   The account which was just deleted.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   *   When affected requests or consumers cannot be deleted or saved.
   */
  protected function handleUserAccountDelete(UserInterface $user): void {
    $requestStorage = $this->entityTypeManager->getStorage('oauth_client_request');
    $requestStorageQuery = $requestStorage->getQuery();
    $ownedRequestIds = $requestStorageQuery->accessCheck(FALSE)->condition('uid', $user->id())->execute();
    $usedRequests = $requestStorageQuery->accessCheck(FALSE)->condition('user', $user->id())->execute();

    // If an owner is deleted, set the request owner to anonymous user.
    if ($ownedRequestIds) {
      foreach ($requestStorage->loadMultiple($ownedRequestIds) as $request) {
        \assert($request instanceof OauthClientRequestInterface);
        $request->setOwnerId(0)->save();
      }
    }

    // If a user is deleted, also delete the request.
    if ($usedRequests) {
      foreach ($requestStorage->loadMultiple($usedRequests) as $request) {
        \assert($request instanceof OauthClientRequestInterface);
        $request->delete();
      }
    }
  }

}
