<?php

declare(strict_types=1);

namespace Drupal\oauth_client\Plugin\Validation\Constraint;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\oauth_client\Entity\OauthClientRequestInterface;
use Drupal\oauth_client\Entity\OauthClientRequestStatus;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\ConstraintValidator;
use Symfony\Component\Validator\Exception\UnexpectedTypeException;

/**
 * Validates the OauthClientRequestTransitionConstraint constraint.
 */
class OauthClientRequestTransitionConstraintValidator extends ConstraintValidator implements ContainerInjectionInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  private $entityTypeManager;

  /**
   * Creates a new OauthClientRequestTransitionConstraintValidator instance.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function validate(mixed $value, Constraint $constraint): void {
    if (!$constraint instanceof OauthClientRequestTransitionConstraint) {
      throw new UnexpectedTypeException($constraint, OauthClientRequestTransitionConstraint::class);
    }

    $entity = $this->context->getObject()->getEntity();

    if (!$entity instanceof OauthClientRequestInterface || $entity->isNew()) {
      return;
    }

    $storage = $this->entityTypeManager->getStorage('oauth_client_request');
    $original = $storage->load($entity->id());

    if (!$original instanceof OauthClientRequestInterface) {
      return;
    }

    $current_status = $original->getStatus();
    $new_status = $value->value;

    if (
      $new_status === OauthClientRequestStatus::Rejected->value &&
      $entity->get('reject_reason')->isEmpty()
    ) {
      $this->context->buildViolation($constraint->rejectEmptyMessage)
        ->addViolation();
    }

    if ($current_status->value === $new_status) {
      return;
    }

    $allowed = OauthClientRequestTransitionHelper::getAllowedTransitions($current_status);

    if (in_array($new_status, $allowed, TRUE)) {
      return;
    }

    $this->context->buildViolation($constraint->transitionMessage)
      ->setParameter('@from', (string) $current_status->value)
      ->setParameter('@to', (string) $new_status)
      ->addViolation();
  }

}
