<?php

declare(strict_types=1);

namespace Drupal\oauth_client\Plugin\views\filter;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\oauth_client\Entity\OauthClientRequestTypeInterface;
use Drupal\simple_oauth\Plugin\ScopeProviderManagerInterface;
use Drupal\views\Attribute\ViewsFilter;
use Drupal\views\Plugin\views\filter\InOperator;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Filter plugin for Oauth2 Client Request's scope (fetched from request type).
 *
 * @ViewsFilter("oauth_client_request_scope")
 */
#[ViewsFilter("oauth_client_request_scope")]
class OauthClientRequestScope extends InOperator implements ContainerFactoryPluginInterface {

  /**
   * Static cache with the scopes per client request type.
   *
   * @var array<string, string>
   */
  protected array $scopePerType;

  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly ScopeProviderManagerInterface $scopeProviderManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('plugin.manager.scope_provider')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getValueOptions(): array {
    // We have to set this property - due to a bad design in the parent class.
    if (isset($this->valueOptions)) {
      return $this->valueOptions;
    }

    $requestTypes = $this->entityTypeManager->getStorage('oauth_client_request_type')->loadMultiple();
    // Keys are the bundle IDs, values are the scopes.
    $this->scopePerType = array_reduce(
      $requestTypes,
      function (array $carry, OauthClientRequestTypeInterface $type): array {
        $carry[$type->id()] = $type->get('scope');
        return $carry;
      },
      [],
    );

    $this->valueOptions = [];
    // We collect all scope, even those which have node request type associated.
    foreach ($this->scopeProviderManager->getInstances() as $provider) {
      foreach ($provider->getScopeProviderAdapter()->loadMultiple() as $scope) {
        $this->valueOptions[$scope->id()] = $scope->getName();
      }
    }
    return $this->valueOptions;
  }

  /**
   * {@inheritdoc}
   */
  public function query(): void {
    if (empty($this->value)) {
      return;
    }
    $this->ensureMyTable();
    if (!$this->query) {
      return;
    }

    $currentScope = is_array($this->value) ? current($this->value) : $this->value;

    // Get types where scope is the value.
    $matchingTypes = array_keys(
      array_filter(
        $this->scopePerType,
        function (string $scope) use ($currentScope): bool {
          return $scope === $currentScope;
        },
      )
    );

    $value = $matchingTypes;
    $operator = $this->operator;
    // If there are no types matching the selected scope we cannot return
    // anything. As a workaround, lets filter for empty type (which cannot be
    // empty actually).
    if (!$matchingTypes) {
      $value = '';
      $operator = '=';
    }

    $this->query->addWhere(
      $this->options['group'],
      "$this->tableAlias.$this->realField",
      $value,
      $operator,
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTags(): array {
    return ['oauth2_scope'];
  }

  /**
   * {@inheritdoc}
   */
  protected function valueForm(&$form, FormStateInterface $form_state): void {
    parent::valueForm($form, $form_state);
    // Not 100% sure about static scopes also utilizing this cache tag.
    $form['#cache']['tags'][] = 'oauth2_scope';
  }

}
