<?php

declare(strict_types=1);

namespace Drupal\oauth_client\Service;

use Drupal\Component\Utility\Crypt;
use Drupal\consumers\Entity\ConsumerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Password\PasswordGeneratorInterface;
use Drupal\oauth_client\Entity\OauthClientRequestInterface;
use Drupal\user\UserInterface;

/**
 * OAuth2 Client Request Consumer service.
 */
class OauthClientConsumerService {

  /**
   * The consumer storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $consumerStorage;

  /**
   * The request type storage.
   *
   * @var \Drupal\oauth_client\Entity\OauthClientRequestTypeInterface
   */
  protected $requestTypeStorage;

  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected PasswordGeneratorInterface $passwordGenerator,
  ) {
    $this->consumerStorage = $entityTypeManager->getStorage('consumer');
    $this->requestTypeStorage = $entityTypeManager->getStorage('oauth_client_request_type');
  }

  /**
   * Creates a Consumer from an OauthClientRequest.
   *
   * @param \Drupal\oauth_client\Entity\OauthClientRequestInterface $clientRequest
   *   The client request.
   *
   * @return \Drupal\consumers\Entity\ConsumerInterface
   *   The consumer.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   *   If consumer cannot be saved.
   * @throws \LogicException
   *   If client request user does not exist.
   */
  public function createConsumerFromRequest(OauthClientRequestInterface $clientRequest): ConsumerInterface {
    $clientId = Crypt::randomBytesBase64();
    $secret = $this->passwordGenerator->generate();
    $requestTypeId = $clientRequest->bundle();
    $requestType = $this->requestTypeStorage->load($requestTypeId);
    $scope_id = $requestType->getScopeId();
    // Filter out the enabled grant types.
    $grantTypes = array_keys(array_filter($requestType->getGrantTypes(), fn($info) => $info['enabled'] ?? FALSE));

    // The intended user of the client must exist, since we handle the delete in
    // OauthClientEntityDeleteHandler::handleUserAccountDelete().
    $clientRequestUser = $clientRequest->getUser();
    if (!$clientRequestUser instanceof UserInterface) {
      throw new \LogicException('Client request user does not exist.');
    }

    /** @var \Drupal\consumers\Entity\ConsumerInterface $consumer */
    $consumer = $this->consumerStorage->create([
      'client_id' => $clientId,
      'owner_id' => $clientRequest->getOwner(),
      'user_id' => $clientRequestUser,
      'label' => $clientRequest->label(),
      'grant_types' => $grantTypes,
      'scopes' => [['scope_id' => $scope_id]],
      'secret' => $secret,
      'confidential' => TRUE,
    ]);
    $consumer->save();

    return $consumer;
  }

}
