<?php

declare(strict_types=1);

namespace Drupal\Tests\oauth_client\Functional;

use Drupal\oauth_client\Entity\OauthClientRequestType;
use Drupal\simple_oauth\Entity\Oauth2Scope;

/**
 * Tests bundle description formatting on the add page.
 *
 * @group oauth_client
 */
class BundleDescriptionFormattingTest extends OauthClientFunctionalTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'block',
    'oauth_client',
    'views',
    'filter',
  ];

  /**
   * Tests that multiple descriptions with various content render correctly.
   */
  public function testMultipleDescriptions(): void {
    // Create a scope for the request types.
    Oauth2Scope::create(['name' => 'test_scope'])->save();

    $grant_type = [
      'client_credentials' => ['enabled' => TRUE, 'settings' => []],
      'authorization_code' => [
        'enabled' => FALSE,
        'settings' => [
          'refresh_token' => TRUE,
          'automatic_authorization' => FALSE,
          'remember_approval' => TRUE,
          'pkce' => FALSE,
        ],
      ],
    ];

    // Create a request type with no description.
    OauthClientRequestType::create([
      'id' => 'no_description',
      'label' => 'No Description Type',
      'description' => '',
      'grant_type' => $grant_type,
      'scope' => 'test_scope',
    ])->save();

    // Create a request type with whitespace-only description.
    OauthClientRequestType::create([
      'id' => 'whitespace',
      'label' => 'Whitespace Type',
      'description' => "   \n\n  ",
      'grant_type' => $grant_type,
      'scope' => 'test_scope',
    ])->save();

    // Create a request type with simple text and link.
    OauthClientRequestType::create([
      'id' => 'with_link',
      'label' => 'With Link Type',
      'description' => 'Request this client.<br>See <a href="https://example.com/docs">documentation</a>.',
      'grant_type' => $grant_type,
      'scope' => 'test_scope',
    ])->save();

    // Create a request type with multiple paragraphs and links.
    OauthClientRequestType::create([
      'id' => 'complex',
      'label' => 'Complex Type',
      'description' => '<p>This provides API access.</p><p>Documentation: <a href="https://example.com/api">API Docs</a> | <a href="https://example.com/support">Support</a></p>',
      'grant_type' => $grant_type,
      'scope' => 'test_scope',
    ])->save();

    // Create a user with permission for all types.
    $user = $this->createUser([
      'request no_description oauth2 client',
      'request whitespace oauth2 client',
      'request with_link oauth2 client',
      'request complex oauth2 client',
    ]);

    // Visit the add page and ensure it loads without errors.
    $this->drupalLogin($user);
    $this->drupalGet("/user/{$user->id()}/edit/oauth/add");
    $this->assertSession()->statusCodeEquals(200);

    // Verify all bundle labels are present.
    $this->assertSession()->pageTextContains('No Description Type');
    $this->assertSession()->pageTextContains('Whitespace Type');
    $this->assertSession()->pageTextContains('With Link Type');
    $this->assertSession()->pageTextContains('Complex Type');

    // Verify formatted descriptions are rendered correctly.
    $this->assertSession()->responseContains('<a href="https://example.com/docs">documentation</a>');
    $this->assertSession()->responseContains('<a href="https://example.com/api">API Docs</a>');
    $this->assertSession()->responseContains('<a href="https://example.com/support">Support</a>');
    $this->assertSession()->pageTextContains('Request this client.');
    $this->assertSession()->pageTextContains('This provides API access.');

    // Verify XSS protection by checking script tags are not rendered.
    $this->assertSession()->responseNotContains('<script>');
  }

}
