<?php

declare(strict_types=1);

namespace Drupal\Tests\oauth_client\Functional;

use Drupal\oauth_client\Entity\OauthClientRequest;
use Drupal\oauth_client\Entity\OauthClientRequestStatus;
use Drupal\oauth_client\Entity\OauthClientRequestType;
use Drupal\oauth_client\Permission\OauthClientPermissions;
use Drupal\simple_oauth\Entity\Oauth2Scope;
use Drupal\user\UserInterface;

/**
 * Functional tests for OAuth Client notifications (mail sending logic).
 *
 * @group oauth_client
 */
class NotificationFunctionalTest extends OauthClientFunctionalTestBase {

  /**
   * The requester user.
   *
   * @var \Drupal\user\UserInterface
   */
  protected UserInterface $requester;

  /**
   * The first manager user.
   *
   * @var \Drupal\user\UserInterface
   */
  protected UserInterface $manager1;

  /**
   * The second manager user.
   *
   * @var \Drupal\user\UserInterface
   */
  protected UserInterface $manager2;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    Oauth2Scope::create(['name' => 'scope1'])->save();
    OauthClientRequestType::create([
      'id' => 'type1',
      'label' => 'Type 1',
      'description' => 'Description of type 1',
      'grant_type' => ['client_credentials' => ['enabled' => TRUE]],
      'scope' => 'scope1',
    ])->save();

    $this->requester = $this->createUser(
      permissions: [OauthClientPermissions::getUserPermission('type1')],
      name: 'requester',
    );
    $this->manager1 = $this->createUser(permissions: ['manage oauth clients'], name: 'manager1');
    $this->manager2 = $this->createUser(permissions: ['manage oauth clients'], name: 'manager2');
  }

  /**
   * Tests that no notifications are sent for disabled actions.
   */
  public function testNoNotificationsWhenDisabled(): void {
    // Disable all notifications.
    $config = $this->container->get('config.factory')->getEditable('oauth_client.settings');
    $config->set('notifications.request.enabled', FALSE);
    $config->set('notifications.approve.enabled', FALSE);
    $config->set('notifications.reject.enabled', FALSE);
    $config->save();

    $requester = user_load_by_name('requester');
    $requesterId = $requester->id();

    // Create two requests by submitting the form as the requester.
    $this->drupalLogin($requester);
    $this->drupalGet("/user/$requesterId/edit/oauth/add");
    $page = $this->getSession()->getPage();
    $page->fillField('Label', 'Test 1');
    $page->fillField('Request reason', 'No notifications expected');
    $page->pressButton('Save');

    $this->drupalGet("/user/$requesterId/edit/oauth/add");
    $page = $this->getSession()->getPage();
    $page->fillField('Label', 'Test 2');
    $page->fillField('Request reason', 'No notifications expected');
    $page->pressButton('Save');

    // Assert no emails were sent (requests are disabled by default).
    $captured = $this->container->get('state')->get('system.test_mail_collector') ?? [];
    $this->assertCount(0, $captured, 'No emails should be sent when request notifications are disabled.');

    // Approve the request via the edit form as a manager.
    $requestId = $this->getRequestByLabel('Test 1')->id();
    $this->drupalLogin(user_load_by_name('manager1'));
    $this->drupalGet("/user/$requesterId/edit/oauth/manage/$requestId");
    $page->selectFieldOption('status', 'active');
    $page->pressButton('Save');

    // Assert no emails were sent (approvals are disabled by default).
    $captured = $this->container->get('state')->get('system.test_mail_collector') ?? [];
    $this->assertCount(0, $captured, 'No emails should be sent when approval notifications are disabled.');

    // Reject the request via the edit form.
    $requestId = $this->getRequestByLabel('Test 2')->id();
    $this->drupalGet("/user/$requesterId/edit/oauth/manage/$requestId");
    $page->selectFieldOption('status', 'rejected');
    $page->fillField('Reject reason', 'Because tests say so');
    $page->pressButton('Save');

    // Assert no emails were sent (rejections are disabled by default).
    $captured = $this->container->get('state')->get('system.test_mail_collector') ?? [];
    $this->assertCount(0, $captured, 'No emails should be sent when rejection notifications are disabled.');
  }

  /**
   * Tests notifications on request creation, approval, and rejection.
   */
  public function testNotificationsOnRequestApproveReject(): void {
    $requesterId = $this->requester->id();

    // Create two requests by the requester via the form.
    $this->drupalLogin($this->requester);

    // First request.
    $this->drupalGet("/user/$requesterId/edit/oauth/add");
    $page = $this->getSession()->getPage();
    $page->fillField('Label', 'First');
    $page->fillField('Request reason', 'I like the first one.');
    $page->pressButton('Save');
    $first = $this->getRequestByLabel('First');
    $firstId = $first->id();
    $this->assertMailsSent(2);

    $this->assertNotificationToEmails($this->manager1, [
      'Dear manager1,',
      'First',
      'I like the first one.',
      // Direct link to the edit form.
      "/user/$requesterId/edit/oauth/manage/$firstId",
      // Part of the management URL.
      '/admin/config/services/consumer/oauth-client-request',
    ]);
    $this->assertNotificationToEmails($this->manager2, [
      'Dear manager2,',
      'First',
      'I like the first one.',
      // Direct link to the edit form.
      "/user/$requesterId/edit/oauth/manage/$firstId",
      // Part of the management URL.
      '/admin/config/services/consumer/oauth-client-request',
    ]);
    // Clear captured emails after assertion to isolate subsequent checks.
    $this->container->get('state')->set('system.test_mail_collector', []);

    // Second request.
    $this->drupalGet("/user/$requesterId/edit/oauth/add");
    $page = $this->getSession()->getPage();
    $page->fillField('Label', 'Second');
    $page->fillField('Request reason', 'I like the second one.');
    $page->pressButton('Save');
    $secondId = $this->getRequestByLabel('Second')->id();
    $this->assertMailsSent(2);

    $this->assertNotificationToEmails($this->manager1, [
      'Dear manager1,',
      'Second',
      'I like the second one.',
      "/user/$requesterId/edit/oauth/manage/$secondId",
      '/admin/config/services/consumer/oauth-client-request',
    ]);

    $this->assertNotificationToEmails($this->manager2, [
      'Dear manager2,',
      'Second',
      'I like the second one.',
      "/user/$requesterId/edit/oauth/manage/$secondId",
      '/admin/config/services/consumer/oauth-client-request',
    ]);
    $this->container->get('state')->set('system.test_mail_collector', []);

    // Approve the first request through the edit form as a manager.
    $this->drupalLogin(user_load_by_name('manager1'));
    $this->drupalGet("/user/$requesterId/edit/oauth/manage/$firstId");
    $page = $this->getSession()->getPage();
    $page->selectFieldOption('status', 'active');
    $page->pressButton('Save');
    // Reload the request to get updated data.
    $first = $this->getRequestByLabel('First');
    $this->assertMailsSent(1);
    $this->assertNotificationToEmails($this->requester, [
      'Dear requester,',
      // The label is mentioned in the email.
      'First',
      // Part of the edit URL.
      "/user/$requesterId/edit/oauth/client/{$first->getClient()->id()}",
      // Part of the user requests URL.
      "/user/$requesterId/edit/oauth",
    ]);
    $this->container->get('state')->set('system.test_mail_collector', []);

    // Reject the second request through the edit form.
    $this->drupalGet("/user/$requesterId/edit/oauth/manage/$secondId");
    $page = $this->getSession()->getPage();
    $page->selectFieldOption('status', 'rejected');
    $page->fillField('Reject reason', 'You are not permitted');
    $page->pressButton('Save');
    $this->assertMailsSent(1);
    $this->assertNotificationToEmails($this->requester, [
      'Dear requester,',
      // The label is mentioned in the email.
      'Second',
      'You are not permitted',
    ]);

    // Assert that the user has access to the client edit URL.
    $this->drupalLogin($this->requester);
    $this->drupalGet("/user/$requesterId/edit/oauth/client/{$first->getClient()->id()}");
    $this->assertSession()->statusCodeEquals(200);
    $this->container->get('state')->set('system.test_mail_collector', []);

    // Assert no email is sent if the request is active already.
    $this->drupalLogin(user_load_by_name('manager1'));

    $this->drupalGet("/user/{$this->manager1->id()}/edit/oauth/add");
    $page = $this->getSession()->getPage();
    $page->fillField('Label', 'NoMail test');
    $page->pressButton('Save');

    $noMailRequest = $this->getRequestByLabel('NoMail test');
    $this->assertSame(OauthClientRequestStatus::Active, $noMailRequest->getStatus());
    $this->assertMailsSent(0);
  }

  /**
   * Returns the latest request created by a given user.
   */
  protected function getRequestByLabel(string $label): ?OauthClientRequest {
    $storage = \Drupal::entityTypeManager()->getStorage('oauth_client_request');
    $ids = \Drupal::entityQuery('oauth_client_request')
      ->accessCheck(FALSE)
      ->condition('label', $label)
      ->range(0, 1)
      ->sort('id', 'DESC')
      ->execute();
    if (!$ids) {
      return NULL;
    }

    // The entity in the test remains in the static cache between the ::get
    // calls and the container in the test, so we need to reset it to ensure we
    // get fresh data from the database.
    $storage->resetCache($ids);
    return OauthClientRequest::load(reset($ids));
  }

  /**
   * Asserts the number of emails sent.
   *
   * @param int $count
   *   The expected number of emails sent.
   */
  protected function assertMailsSent(int $count): void {
    $captured = $this->container->get('state')->get('system.test_mail_collector') ?? [];
    $this->assertCount($count, $captured, 'Expected number of emails sent.');
  }

  /**
   * Asserts that notification emails were sent to the expected addresses.
   *
   * @param \Drupal\user\UserInterface $recipient
   *   The recipient of the email.
   * @param string[] $textParts
   *   The expected text parts that should be contained in each email body.
   */
  protected function assertNotificationToEmails(UserInterface $recipient, array $textParts = []): void {
    $captured = $this->container->get('state')->get('system.test_mail_collector') ?? [];
    $sentTo = array_map(fn($m) => $m['to'] ?? '', $captured);
    $this->assertTrue(in_array($recipient->getEmail(), $sentTo), 'Notification emails were sent to the expected addresses.');

    foreach ($captured as $mail) {
      if ($mail['to'] !== $recipient->getEmail()) {
        continue;
      }

      foreach ($textParts as $textPart) {
        $this->assertStringContainsString($textPart, $mail['body']);
      }
    }
  }

}
