<?php

declare(strict_types=1);

namespace Drupal\Tests\oauth_client\Functional;

use Drupal\filter\Entity\FilterFormat;

/**
 * Tests oauth_client_minimal text format.
 *
 * @group oauth_client
 */
class TextFormatAccessTest extends OauthClientFunctionalTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'block',
    'oauth_client',
    'views',
    'filter',
  ];

  /**
   * The filter format entity.
   *
   * @var \Drupal\filter\Entity\FilterFormat
   */
  protected FilterFormat $format;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->format = FilterFormat::load('oauth_client_minimal');
    $this->assertNotNull($this->format);
  }

  /**
   * Tests format metadata and configuration.
   */
  public function testFormatConfiguration(): void {
    $this->assertEquals('oauth_client_minimal', $this->format->id());
    $this->assertEquals('OAuth Client Minimal (Internal)', $this->format->label());

    $filters = $this->format->get('filters');
    $this->assertTrue($filters['filter_html']['status']);
    $this->assertTrue($filters['filter_autop']['status']);
    $this->assertTrue($filters['filter_url']['status']);

    $allowed_html = $filters['filter_html']['settings']['allowed_html'];
    $this->assertStringContainsString('<a href hreflang>', $allowed_html);
    $this->assertStringContainsString('<p>', $allowed_html);
    $this->assertStringContainsString('<br>', $allowed_html);
  }

  /**
   * Tests line break and paragraph conversion.
   */
  public function testTextFormatting(): void {
    $text = "Single line\nbreak\n\nDouble line\nbreak";
    $processed = (string) check_markup($text, 'oauth_client_minimal');

    $this->assertStringContainsString('<br', $processed);
    $this->assertStringContainsString('</p>', $processed);
  }

  /**
   * Tests URL conversion to links.
   */
  public function testUrlConversion(): void {
    $text = 'Visit https://example.com and www.test.com';
    $processed = (string) check_markup($text, 'oauth_client_minimal');

    $this->assertStringContainsString('<a href="https://example.com">https://example.com</a>', $processed);
    $this->assertStringContainsString('<a href="http://www.test.com">www.test.com</a>', $processed);
  }

  /**
   * Tests XSS prevention.
   */
  public function testXssPrevention(): void {
    $malicious = 'Text with <script>alert("xss")</script> and <img src=x onerror=alert(1)>';
    $processed = (string) check_markup($malicious, 'oauth_client_minimal');

    $this->assertStringNotContainsString('<script>', $processed);
    $this->assertStringNotContainsString('<img', $processed);
    $this->assertStringNotContainsString('onerror', $processed);
  }

  /**
   * Tests complex content with URLs, paragraphs, and line breaks.
   */
  public function testComplexContent(): void {
    $text = "Paragraph with https://example.com link.\n\nSecond paragraph\nwith line break.";
    $processed = (string) check_markup($text, 'oauth_client_minimal');

    $this->assertStringContainsString('<p>', $processed);
    $this->assertStringContainsString('<br', $processed);
    $this->assertStringContainsString('<a href="https://example.com">https://example.com</a>', $processed);
  }

  /**
   * Tests internal use indication and access control.
   */
  public function testInternalUseIndication(): void {
    $this->assertStringContainsString('(Internal)', (string) $this->format->label());

    $roles = filter_get_roles_by_format($this->format);
    $this->assertEmpty($roles);
  }

}
