<?php

declare(strict_types=1);

namespace src\Kernel;

use Drupal\Core\Url;
use Drupal\KernelTests\KernelTestBase;
use Drupal\oauth_client\Entity\OauthClientRequest;
use Drupal\oauth_client\Entity\OauthClientRequestStatus;
use Drupal\oauth_client\Entity\OauthClientRequestType;
use Drupal\simple_oauth\Entity\Oauth2Scope;
use Drupal\Tests\user\Traits\UserCreationTrait;

/**
 * Kernel tests verifying oauth_client tokens and token info.
 *
 * @group oauth_client
 */
class TokensKernelTest extends KernelTestBase {

  use UserCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'consumers',
    'file',
    'filter',
    'image',
    'oauth_client',
    'options',
    'serialization',
    'simple_oauth',
    'system',
    'user',
    'views',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('consumer');
    $this->installEntitySchema('oauth2_scope');
    $this->installEntitySchema('user');
    $this->installEntitySchema('oauth_client_request');
    $this->installConfig(['oauth_client', 'simple_oauth', 'user', 'system']);

    // Seed a scope and a request type to be able to create requests.
    Oauth2Scope::create(['name' => 'scopeT'])->save();
    OauthClientRequestType::create([
      'id' => 'typeT',
      'label' => 'Type T',
      'grant_type' => ['client_credentials' => ['enabled' => TRUE]],
      'scope' => 'scopeT',
    ])->save();
  }

  /**
   * Tests token replacement for oauth_client module.
   */
  public function testTokenReplacement(): void {
    // Create a user who will own the request.
    $account = $this->createUser([], 'token_user', FALSE, ['mail' => 'token_user@example.com']);

    // Create a request and set request/reject reasons and timestamps.
    $created = 1700000000;
    $changed = 1700003600;
    $request = OauthClientRequest::create([
      'type' => 'typeT',
      'label' => 'My Label',
      'uid' => $account,
      'user' => $account,
      'status' => OauthClientRequestStatus::Pending->value,
      'request_reason' => 'Because I need it',
      'reject_reason' => 'Not applicable',
      'created' => $created,
      'changed' => $changed,
    ]);
    $request->save();

    $formatter = $this->container->get('date.formatter');
    $expectedCreated = $formatter->format($created, 'medium', '', NULL);
    $expectedChanged = $formatter->format($changed, 'medium', '', NULL);

    $token = $this->container->get('token');

    // Replace individual tokens.
    $data = [
      'oauth_client_request' => $request,
      'user' => $account,
      'oauth_client' => NULL,
    ];

    $this->assertSame('My Label', $token->replace('[oauth_client_request:label]', $data, ['clear' => TRUE]));
    $this->assertSame('Because I need it', $token->replace('[oauth_client_request:request_reason]', $data, ['clear' => TRUE]));
    $this->assertSame('Not applicable', $token->replace('[oauth_client_request:reject_reason]', $data, ['clear' => TRUE]));

    // Owner should resolve to the account label (username).
    $this->assertSame($account->label(), $token->replace('[oauth_client_request:owner]', $data, ['clear' => TRUE]));

    // Created/changed should use a medium date format.
    $this->assertSame($expectedCreated, $token->replace('[oauth_client_request:created]', $data, ['clear' => TRUE]));
    $this->assertSame($expectedChanged, $token->replace('[oauth_client_request:changed]', $data, ['clear' => TRUE]));

    $editUrl = $token->replace('[oauth_client_request:request-edit-url]', $data, ['clear' => TRUE]);
    $this->assertSame($editUrl, $request->toUrl('edit-form')->setAbsolute()->toString());

    $request->setStatus(OauthClientRequestStatus::Active)->save();
    $editUrl = $token->replace('[oauth_client_request:client-edit-url]', $data, ['clear' => TRUE]);
    $this->assertSame($editUrl, Url::fromRoute('oauth_client.edit', [
      'user' => $account->id(),
      'client' => $request->getClient()->id(),
    ])->setAbsolute()->toString());

    // Test user:oauth_client_requests URL generation.
    $userRequestsUrl = $token->replace('[oauth_client_request:owner:client_requests]', $data, ['clear' => TRUE]);
    $expectedUserRequestsUrl = Url::fromRoute('view.oauth_client_request.user_requests', ['user' => $account->id()])->setAbsolute()->toString();
    $this->assertSame($expectedUserRequestsUrl, $userRequestsUrl);

    // Test oauth_client:client_requests URL generation.
    $url = $token->replace('[oauth_client:client_requests]', [], ['clear' => TRUE]);
    $viewUrl = Url::fromRoute('view.oauth_client_request.admin_page')->setAbsolute()->toString();
    $this->assertSame($viewUrl, $url);
  }

}
