<?php

declare(strict_types=1);

namespace Drupal\oauth_client\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\DependencyInjection\ClassResolverInterface;
use Drupal\Core\Entity\Controller\EntityController;
use Drupal\oauth_client\Access\AccessTrait;
use Drupal\user\UserInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Request a new OAuth2 client.
 */
class OauthClientRequestAddController extends ControllerBase {

  use AccessTrait;

  /**
   * The decorated entity controller.
   */
  protected EntityController $original;

  public function __construct(
    protected readonly ClassResolverInterface $classResolver,
  ) {
    $this->original = $this->classResolver->getInstanceFromDefinition(EntityController::class);
  }

  /**
   * Renders the 'add-page' page.
   */
  public function addPage($entity_type_id, ?UserInterface $user = NULL): RedirectResponse|array {
    $response = $this->original->addPage($entity_type_id);

    if (!is_array($response)) {
      // It's a redirect.
      return $response;
    }

    // Add the missing route parameter.
    foreach ($response['#bundles'] as $bundle) {
      $bundle['add_link']->getUrl()->setRouteParameter('user', $user->id());
    }

    return $response;
  }

  /**
   * Provides a title callback for 'add-page' route.
   */
  public function addTitle(?UserInterface $user = NULL): string|\Stringable {
    $currentUser = $this->currentUser();

    if (!$this->isManager($currentUser)) {
      return $this->t('Request a new client');
    }

    if ($user->id() != $currentUser->id()) {
      return $this->t('Create a new client for %user', [
        '%user' => $user->getDisplayName(),
      ]);
    }

    return $this->t('Create a new client');
  }

}
