<?php

declare(strict_types=1);

namespace Drupal\oauth_client\Entity;

use Drupal\consumers\Entity\ConsumerInterface;
use Drupal\Core\Entity\Attribute\ContentEntityType;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\ContentEntityDeleteForm;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\oauth_client\Access\OauthClientRequestAccessControlHandler;
use Drupal\oauth_client\Form\OauthClientRequestForm;
use Drupal\oauth_client\Routing\OauthClientRequestHtmlRouteProvider;
use Drupal\oauth_client\OauthClientRequestViewsData;
use Drupal\simple_oauth\Entity\Oauth2ScopeEntityInterface;
use Drupal\user\EntityOwnerTrait;
use Drupal\user\UserInterface;

/**
 * Defines the OAuth Client Request entity.
 *
 * @ContentEntityType(
 *   id = "oauth_client_request",
 *   label = @Translation("OAuth2 Client Request"),
 *   label_collection = @Translation("OAuth2 Client Requests"),
 *   label_singular = @Translation("OAuth2 client request"),
 *   label_plural = @Translation("OAuth2 client requests"),
 *   entity_keys = {
 *     "id" = "id",
 *     "bundle" = "type",
 *     "label" = "label",
 *     "owner" = "uid",
 *   },
 *   handlers = {
 *     "access" = "Drupal\oauth_client\Access\OauthClientRequestAccessControlHandler",
 *     "form" = {
 *       "default" = "Drupal\oauth_client\Form\OauthClientRequestForm",
 *       "delete" = "Drupal\Core\Entity\ContentEntityDeleteForm",
 *     },
 *     "list_builder" = "Drupal\Core\Entity\EntityListBuilder",
 *     "route_provider" = {
 *       "html" = "Drupal\oauth_client\Routing\OauthClientRequestHtmlRouteProvider",
 *     },
 *     "views_data" = "Drupal\oauth_client\OauthClientRequestViewsData",
 *   },
 *   links = {
 *     "collection" = "/admin/config/services/consumer/oauth-client-request",
 *     "add-page" = "/user/{user}/edit/oauth/add",
 *     "add-form" = "/user/{user}/edit/oauth/add/{oauth_client_request_type}",
 *     "canonical" = "/user/{user}/edit/oauth/manage/{oauth_client_request}",
 *     "delete-form" = "/user/{user}/edit/oauth/delete/{oauth_client_request}",
 *     "edit-form" = "/user/{user}/edit/oauth/manage/{oauth_client_request}",
 *   },
 *   admin_permission = "manage oauth clients",
 *   permission_granularity = "bundle",
 *   bundle_entity_type = "oauth_client_request_type",
 *   bundle_label = @Translation("OAuth2 Client Request Type"),
 *   base_table = "oauth_client_request",
 *   translatable = FALSE,
 *   label_count = {
 *     "singular" = "@count OAuth2 client request",
 *     "plural" = "@count OAuth2 client requests",
 *   },
 *   field_ui_base_route = "entity.oauth_client_request_type.canonical",
 * )
 *
 * @phpstan-ignore-next-line Drupal\Core\Entity\Attribute\ContentEntityType
 */
#[ContentEntityType(
  id: 'oauth_client_request',
  label: new TranslatableMarkup('OAuth2 Client Request'),
  label_collection: new TranslatableMarkup('OAuth2 Client Requests'),
  label_singular: new TranslatableMarkup('OAuth2 client request'),
  label_plural: new TranslatableMarkup('OAuth2 client requests'),
  entity_keys: [
    'id' => 'id',
    'bundle' => 'type',
    'label' => 'label',
    'owner' => 'uid',
  ],
  handlers: [
    'access' => OauthClientRequestAccessControlHandler::class,
    'form' => [
      'default' => OauthClientRequestForm::class,
      'delete' => ContentEntityDeleteForm::class,
    ],
    'list_builder' => EntityListBuilder::class,
    'route_provider' => [
      'html' => OauthClientRequestHtmlRouteProvider::class,
    ],
    'views_data' => OauthClientRequestViewsData::class,
  ],
  links: [
    'collection' => '/admin/config/services/consumer/oauth-client-request',
    'add-page' => '/user/{user}/edit/oauth/add',
    'add-form' => '/user/{user}/edit/oauth/add/{oauth_client_request_type}',
    'canonical' => '/user/{user}/edit/oauth/manage/{oauth_client_request}',
    'delete-form' => '/user/{user}/edit/oauth/delete/{oauth_client_request}',
    'edit-form' => '/user/{user}/edit/oauth/manage/{oauth_client_request}',
  ],
  admin_permission: 'manage oauth clients',
  permission_granularity: 'bundle',
  bundle_entity_type: 'oauth_client_request_type',
  bundle_label: new TranslatableMarkup('OAuth2 Client Request Type'),
  base_table: 'oauth_client_request',
  translatable: FALSE,
  label_count: [
    'singular' => '@count OAuth2 client request',
    'plural' => '@count OAuth2 client requests',
  ],
  field_ui_base_route: 'entity.oauth_client_request_type.canonical',
)]
class OauthClientRequest extends ContentEntityBase implements OauthClientRequestInterface {

  use EntityChangedTrait;
  use EntityOwnerTrait;

  /**
   * {@inheritdoc}
   */
  public function setLabel(string $label): OauthClientRequestInterface {
    $this->set('label', $label);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getClient(): ?ConsumerInterface {
    return $this->get('client')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function setClient(?ConsumerInterface $client): OauthClientRequestInterface {
    $this->set('client', $client);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getUser(): UserInterface {
    return $this->get('user')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function setUser(UserInterface $user): OauthClientRequestInterface {
    $this->set('user', $user);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getScope(): ?Oauth2ScopeEntityInterface {
    return $this->get('scope')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function setScope(?Oauth2ScopeEntityInterface $scope): OauthClientRequestInterface {
    $this->set('scope', $scope);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getStatus(): OauthClientRequestStatus {
    $value = $this->get('status')->value ?? OauthClientRequestStatus::Pending->value;
    return OauthClientRequestStatus::from($value);
  }

  /**
   * {@inheritdoc}
   */
  public function setStatus(OauthClientRequestStatus $status): OauthClientRequestInterface {
    $this->set('status', $status->value);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function isStatus(OauthClientRequestStatus $status): bool {
    return $this->getStatus() === $status;
  }

  /**
   * {@inheritdoc}
   */
  public function getRequestReason(): ?string {
    return $this->get('request_reason')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setRequestReason(?string $reason): OauthClientRequestInterface {
    $this->set('request_reason', $reason);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getRejectReason(): ?string {
    return $this->get('reject_reason')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setRejectReason(?string $reason): OauthClientRequestInterface {
    $this->set('reject_reason', $reason);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getCreatedTime(): int {
    return (int) $this->get('created')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setCreatedTime(int $timestamp): OauthClientRequestInterface {
    $this->set('created', $timestamp);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  protected function urlRouteParameters($rel): array {
    $uri_route_parameters = parent::urlRouteParameters($rel);

    // Add the user parameter for all routes.
    if ($this->getUser()) {
      $uri_route_parameters['user'] = $this->getUser()->id();
    }

    return $uri_route_parameters;
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {
    $fields = parent::baseFieldDefinitions($entity_type);

    // Override the default created by entity owner trait.
    $fields += static::ownerBaseFieldDefinitions($entity_type);

    $fields['label'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Label'))
      ->setDescription(t('The preferred consumer label. Will appear in the user screen.'))
      ->setRequired(TRUE)
      ->setSetting('max_length', 255)
      ->setDisplayOptions(
        'view', [
          'label' => 'above',
          'type' => 'string',
          'weight' => -4,
        ]
      )
      ->setDisplayOptions(
        'form', [
          'type' => 'string_textfield',
          'weight' => -4,
        ]
      )
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['client'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Client'))
      ->setDescription(t('The associated consumer entity.'))
      ->setSetting('target_type', 'consumer')
      ->setCardinality(1)
      ->setDisplayOptions(
        'view', [
          'label' => 'above',
          'type' => 'entity_reference_label',
          'weight' => 0,
        ]
      )
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['user'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('User'))
      ->setDescription(t('The user who will own/use the client.'))
      ->setSetting('target_type', 'user')
      ->setRequired(TRUE)
      ->setCardinality(1)
      ->setDisplayOptions(
        'view', [
          'label' => 'above',
          'type' => 'entity_reference_label',
          'weight' => 0,
        ]
      )
      ->setDisplayOptions(
        'form', [
          'type' => 'entity_reference_autocomplete',
          'weight' => 0,
        ]
      )
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['request_reason'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('Request reason'))
      ->setDescription(t('The text the user fills in when requesting the client.'))
      ->setDisplayOptions(
        'view', [
          'label' => 'above',
          'type' => 'basic_string',
          'weight' => 0,
          'settings' => [
            'format' => 'plain_text',
          ],
        ]
      )
      ->setDisplayOptions(
        'form', [
          'type' => 'string_textarea',
          'weight' => 0,
        ]
      )
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['status'] = BaseFieldDefinition::create('list_string')
      ->setLabel(t('Status'))
      ->setRequired(TRUE)
      ->setDefaultValue(OauthClientRequestStatus::Pending->value)
      ->setSetting('allowed_values', OauthClientRequestStatus::getOptions())
      ->setDisplayOptions(
        'view', [
          'label' => 'above',
          'type' => 'list_default',
          'weight' => 0,
        ]
      )
      ->setDisplayOptions(
        'form', [
          'type' => 'options_buttons',
          'weight' => 0,
        ]
      )
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['reject_reason'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('Reject reason'))
      ->setDescription(t('The reason why the request was rejected.'))
      ->setDisplayOptions(
        'view', [
          'label' => 'above',
          'type' => 'basic_string',
          'weight' => 0,
          'settings' => [
            'format' => 'plain_text',
          ],
        ]
      )
      ->setDisplayOptions(
        'form', [
          'type' => 'string_textarea',
          'weight' => 0,
        ]
      )
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['redirect'] = BaseFieldDefinition::create('uri')
      ->setLabel(t('Redirect URI'))
      ->setCardinality(FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED)
      ->setDisplayOptions(
        'view', [
          'label' => 'above',
          'type' => 'uri_link',
          'weight' => 0,
        ]
      )
      ->setDisplayOptions(
        'form', [
          'type' => 'uri',
          'weight' => 0,
        ]
      )
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the OAuth Client Request was created.'));

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the OAuth Client Request was last edited.'));

    return $fields;
  }

}
