<?php

declare(strict_types=1);

namespace Drupal\oauth_client\Entity;

use Drupal\Core\Config\Entity\ConfigEntityBundleBase;
use Drupal\Core\Entity\Attribute\ConfigEntityType;
use Drupal\Core\Entity\EntityAccessControlHandler;
use Drupal\Core\Entity\EntityDeleteForm;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\oauth_client\Form\OauthClientRequestTypeForm;
use Drupal\oauth_client\Routing\OauthClientRequestTypeHtmlRouteProvider;
use Drupal\simple_oauth\Entity\Oauth2ScopeEntityInterface;
use Drupal\simple_oauth\Oauth2ScopeInterface;
use Drupal\user\Entity\EntityPermissionsRouteProvider;

/**
 * Defines the OAuth2 Client Request Type entity.
 *
 * @ConfigEntityType(
 *   id = "oauth_client_request_type",
 *   label = @Translation("OAuth2 Client Request Type"),
 *   label_collection = @Translation("OAuth2 Client Request Types"),
 *   label_singular = @Translation("OAuth2 client request type"),
 *   label_plural = @Translation("OAuth2 client requests types"),
 *   config_prefix = "type",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *   },
 *   handlers = {
 *     "access" = "Drupal\Core\Entity\EntityAccessControlHandler",
 *     "form" = {
 *       "default" = "Drupal\oauth_client\Form\OauthClientRequestTypeForm",
 *       "delete" = "Drupal\Core\Entity\EntityDeleteForm",
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\oauth_client\Routing\OauthClientRequestTypeHtmlRouteProvider",
 *       "permissions" = "Drupal\user\Entity\EntityPermissionsRouteProvider",
 *     },
 *     "list_builder" = "Drupal\oauth_client\Entity\OauthClientTypeListBuilder",
 *   },
 *   links = {
 *     "collection" = "/admin/config/services/consumer/oauth-client",
 *     "canonical" = "/admin/config/services/consumer/oauth-client/{oauth_client_request_type}",
 *     "add-form" = "/admin/config/services/consumer/oauth-client/add",
 *     "delete-form" = "/admin/config/services/consumer/oauth-client/{oauth_client_request_type}/delete",
 *     "entity-permissions-form" = "/admin/config/services/consumer/oauth-client/{oauth_client_request_type}/permissions",
 *   },
 *   admin_permission = "administer oauth clients",
 *   bundle_of = "oauth_client_request",
 *   label_count = {
 *     "singular" = "@count OAuth2 client request type",
 *     "plural" = "@count OAuth2 client request types",
 *   },
 *   config_export = {
 *     "id",
 *     "label",
 *     "description",
 *     "grant_type",
 *     "scope",
 *   },
 * )
 *
 * @phpstan-ignore-next-line Drupal\Core\Entity\Attribute\ConfigEntityType
 */
#[ConfigEntityType(
  id: 'oauth_client_request_type',
  label: new TranslatableMarkup('OAuth2 Client Request Type'),
  label_collection: new TranslatableMarkup('OAuth2 Client Request Types'),
  label_singular: new TranslatableMarkup('OAuth2 client request type'),
  label_plural: new TranslatableMarkup('OAuth2 client requests types'),
  config_prefix: 'type',
  entity_keys: [
    'id' => 'id',
    'label' => 'label',
  ],
  handlers: [
    'access' => EntityAccessControlHandler::class,
    'form' => [
      'default' => OauthClientRequestTypeForm::class,
      'delete' => EntityDeleteForm::class,
    ],
    'route_provider' => [
      'html' => OauthClientRequestTypeHtmlRouteProvider::class,
      'permissions' => EntityPermissionsRouteProvider::class,
    ],
    'list_builder' => OauthClientTypeListBuilder::class,
  ],
  links: [
    'collection' => '/admin/config/services/consumer/oauth-client',
    'canonical' => '/admin/config/services/consumer/oauth-client/{oauth_client_request_type}',
    'add-form' => '/admin/config/services/consumer/oauth-client/add',
    'delete-form' => '/admin/config/services/consumer/oauth-client/{oauth_client_request_type}/delete',
    'entity-permissions-form' => '/admin/config/services/consumer/oauth-client/{oauth_client_request_type}/permissions',
  ],
  admin_permission: 'administer oauth clients',
  bundle_of: 'oauth_client_request',
  label_count: [
    'singular' => '@count OAuth2 client request type',
    'plural' => '@count OAuth2 client request types',
  ],
  config_export: [
    'id',
    'label',
    'description',
    'grant_type',
    'scope',
  ],
)]
class OauthClientRequestType extends ConfigEntityBundleBase implements OauthClientRequestTypeInterface {

  /**
   * The OAuth2 client request type ID.
   */
  protected string $id;

  /**
   * The OAuth2 client request type label.
   */
  protected string|\Stringable $label;

  /**
   * The OAuth2 client request type label.
   */
  protected string|\Stringable $description = '';

  /**
   * The OAuth2 grant type: 'client_credentials' or 'authorization_code'.
   *
   * @var array<array-key, array>
   */
  protected array $grant_type = [
    'client_credentials' => ['enabled' => FALSE, 'settings' => []],
    'authorization_code' => [
      'enabled' => FALSE,
      'settings' => [
        'refresh_token' => TRUE,
        'automatic_authorization' => FALSE,
        'remember_approval' => TRUE,
        'pkce' => FALSE,
      ],
    ],
  ];

  /**
   * The OAuth2 scope.
   */
  protected ?string $scope = NULL;

  /**
   * {@inheritdoc}
   */
  public function getDescription(): string|\Stringable {
    return $this->description;
  }

  /**
   * {@inheritdoc}
   */
  public function setDescription($description): void {
    $this->description = $description;
  }

  /**
   * {@inheritdoc}
   */
  public function getGrantTypes(): array {
    return $this->grant_type;
  }

  /**
   * {@inheritdoc}
   */
  public function isEnabledGrantType(GrantType $grantType): bool {
    return $this->getGrantTypes()[$grantType->value]['enabled'];
  }

  /**
   * {@inheritdoc}
   */
  public function getGrantTypeSetting(GrantType $grantType, string $key): mixed {
    return $this->getGrantTypes()[$grantType->value]['settings'][$key];
  }

  /**
   * {@inheritdoc}
   */
  public function getScopeId(): ?string {
    return $this->scope;
  }

  /**
   * {@inheritdoc}
   */
  public function getScope(): ?Oauth2ScopeInterface {
    if ($scopeId = $this->getScopeId()) {
      return \Drupal::service('simple_oauth.oauth2_scope.provider')->load($scopeId);
    }
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function calculateDependencies(): static {
    parent::calculateDependencies();

    $scope = $this->getScope();
    if ($scope instanceof Oauth2ScopeEntityInterface) {
      $this->addDependency($scope->getConfigDependencyKey(), $scope->getConfigDependencyName());
    }

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function onDependencyRemoval(array $dependencies): bool {
    $changed = parent::onDependencyRemoval($dependencies);

    foreach ($dependencies['config'] as $entity) {
      if (!$entity instanceof Oauth2ScopeEntityInterface) {
        continue;
      }

      $hasContent = (bool) \Drupal::entityQuery('oauth_client_request')
        ->accessCheck(FALSE)
        ->count()
        ->condition('type', $this->id())
        ->execute();
      if ($hasContent) {
        // If there are existing requests with this bundle, we cannot let the
        // bundle entity to be removed. Instead, we unset the scope letting the
        // bundle alive. In this case, new content cannot be created until the
        // bundle is manually fixed by setting a new scope. If there are no
        // existing requests having this bundle, the bundle is safely deleted.
        // @see \Drupal\oauth_client\Service\OauthClientRequestTypeHelper::isOauthClientRequestTypeValid()
        // @see \Drupal\oauth_client\Access\OauthClientRequestAccessControlHandler::checkCreateAccess()
        $this->set('scope', NULL);
        $changed = TRUE;
        // Once this entity needs save we don't check other dependencies.
        break;
      }
    }

    return $changed;
  }

}
