<?php

declare(strict_types=1);

namespace Drupal\oauth_client\Entity;

use Drupal\Core\Config\Entity\ConfigEntityInterface;
use Drupal\Core\Entity\EntityDescriptionInterface;
use Drupal\simple_oauth\Oauth2ScopeInterface;

/**
 * Interface for the OAuth2 Client Request Type entity.
 */
interface OauthClientRequestTypeInterface extends ConfigEntityInterface, EntityDescriptionInterface {

  /**
   * Returns a list of OAuth2 grant types.
   *
   * @return array<array-key, array{enabled: bool, settings: array}>
   *   The OAuth2 grant types. Keys are client_credentials, authorization_code.
   *   Values are associative arrays:
   *   - enabled (boolean): If the grant type is enabled.
   *   - settings (array): Grant type settings.
   */
  public function getGrantTypes(): array;

  /**
   * Checks whether a grant type is enabled.
   *
   * @param \Drupal\oauth_client\Entity\GrantType $grantType
   *   The grant type.
   *
   * @return bool
   *   Whether a grant type is enabled.
   */
  public function isEnabledGrantType(GrantType $grantType): bool;

  /**
   * Returns the settings specific to a grant type.
   *
   * @param \Drupal\oauth_client\Entity\GrantType $grantType
   *   The grant type.
   * @param string $key
   *   The key of the setting.
   *
   * @return mixed
   *   Setting to be retrieved.
   */
  public function getGrantTypeSetting(GrantType $grantType, string $key): mixed;

  /**
   * Returns the OAuth2 scope ID.
   *
   * @return string|null
   *   The OAuth2 scope ID.
   */
  public function getScopeId(): ?string;

  /**
   * Returns the OAuth2 scope.
   *
   * @return \Drupal\simple_oauth\Oauth2ScopeInterface|null
   *   The OAuth2 scope.
   */
  public function getScope(): ?Oauth2ScopeInterface;

}
