<?php

declare(strict_types=1);

namespace Drupal\oauth_client\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure OAuth2 Client notification settings.
 */
final class NotificationSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'oauth_client_notification_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['oauth_client.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['#tree'] = TRUE;
    $form['notifications']['request'] = $this->buildActionElement('request', 'Request');
    $form['notifications']['approve'] = $this->buildActionElement('approve', 'Approve');
    $form['notifications']['reject'] = $this->buildActionElement('reject', 'Reject');

    return parent::buildForm($form, $form_state);
  }

  /**
   * Helper to build a notification sub-form.
   *
   * @param string $key
   *   The notification key.
   * @param string $title
   *   The notification title.
   *
   * @return array
   *   The form element.
   */
  protected function buildActionElement(string $key, string $title): array {
    $element = [
      '#type' => 'details',
      '#title' => $this->t('@title notification', ['@title' => $title]),
      '#open' => TRUE,
    ];
    $element['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enabled'),
      '#config_target' => "oauth_client.settings:notifications.$key.enabled",
    ];
    $element['subject'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Subject'),
      '#config_target' => "oauth_client.settings:notifications.$key.subject",
      '#required' => TRUE,
    ];
    $element['body'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Body'),
      '#config_target' => "oauth_client.settings:notifications.$key.body",
      '#required' => TRUE,
    ];

    $element['help_text'] = [
      '#type' => 'item',
      // We cannot use the tokens service as we would need to inject the
      // service, and thus, override the constructor. This will result in
      // tests failing as long as the previous major version is part of the
      // tests. The reason is that there has been a change in the signature
      // of the parent constructor between Drupal 10 and 11.
      // @see https://www.drupal.org/node/3404140
      '#markup' => $this->t('Tokens are supported. You may use the following tokens: <br>
        <code>[current-user:display-name]</code>,
        <code>[current-user:mail]</code>,
        <code>[oauth_client_request:label</code>,
        <code>[oauth_client_request:request_reason]</code> etc.'),
    ];

    return $element;
  }

}
