<?php

declare(strict_types=1);

namespace Drupal\oauth_client\Form;

use Drupal\consumers\Entity\ConsumerInterface;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\user\UserInterface;

/**
 * Form for deleting an OAuth2 client.
 */
class OauthClientDeleteForm extends ConfirmFormBase {

  use FormAccessTrait;

  /**
   * The user account.
   *
   * @var \Drupal\user\UserInterface
   */
  protected UserInterface $user;

  /**
   * The OAuth2 client.
   *
   * @var \Drupal\consumers\Entity\ConsumerInterface
   */
  protected ConsumerInterface $client;

  /**
   * {@inheritdoc}
   */
  public function getQuestion(): TranslatableMarkup {
    return $this->t('Are you sure you want to delete the %client OAuth2 client?', [
      '%client' => $this->client->label(),
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl(): Url {
    return new Url('oauth_client.user', ['user' => $this->user->id()]);
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?UserInterface $user = NULL, ?ConsumerInterface $client = NULL): array {
    $this->user = $user;
    $this->client = $client;
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->client->delete();
    $this->messenger()
      ->addStatus($this->t('The %client OAuth2 client has been deleted.', [
        '%client' => $this->client->label(),
      ]));
    $form_state->setRedirect('oauth_client.user', ['user' => $this->user->id()]);
  }

  /**
   * Checks access to the form.
   *
   * @param \Drupal\user\UserInterface $user
   *   The user account.
   * @param \Drupal\consumers\Entity\ConsumerInterface $client
   *   The OAuth2 client.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result object.
   */
  public function access(UserInterface $user, ConsumerInterface $client): AccessResultInterface {
    return $this->checkAccess('delete', $user, $client);
  }

  /**
   * Returns the title of the form.
   *
   * @param \Drupal\consumers\Entity\ConsumerInterface $client
   *   The OAuth2 client.
   */
  public function title(ConsumerInterface $client): TranslatableMarkup {
    return $this->t('Delete the %client OAuth2 client', ['%client' => $client->label()]);
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'oauth_client_delete';
  }

}
