<?php

declare(strict_types=1);

namespace Drupal\oauth_client\Hook;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\oauth_client\Entity\OauthClientRequestInterface;
use Drupal\oauth_client\Entity\OauthClientRequestStatus;
use Drupal\oauth_client\Service\OauthClientConsumerService;

/**
 * Create client consumer on request insert.
 */
class OauthClientConsumerHooks {

  public function __construct(
    protected OauthClientConsumerService $oauthClientConsumerService,
  ) {}

  /**
   * Acts after creating an oauth_client_request entity.
   *
   * @phpstan-ignore-next-line as Attribute class Drupal\Core\Hook\Attribute\Hook does not exist.
   */
  #[Hook('oauth_client_request_insert')]
  public function onInsert(EntityInterface $entity): void {
    if (!$entity instanceof OauthClientRequestInterface || $entity->getClient()) {
      return;
    }

    if ($entity->getStatus() !== OauthClientRequestStatus::Active) {
      return;
    }

    $consumer = $this->oauthClientConsumerService->createConsumerFromRequest($entity);
    $entity->setClient($consumer);
    $entity->save();
  }

  /**
   * Acts after updating an oauth_client_request entity.
   *
   * @phpstan-ignore-next-line as Attribute class Drupal\Core\Hook\Attribute\Hook does not exist.
   */
  #[Hook('oauth_client_request_update')]
  public function onUpdate(EntityInterface $entity): void {
    if (!$entity instanceof OauthClientRequestInterface || $entity->getClient()) {
      return;
    }

    $original = $entity->original;

    if (!$original instanceof OauthClientRequestInterface) {
      return;
    }

    // Check if the status changed from pending or rejected to active.
    $previousStatuses = [OauthClientRequestStatus::Pending, OauthClientRequestStatus::Rejected];
    $isPendingOrRejected = in_array($original->getStatus(), $previousStatuses);
    $isActive = $entity->getStatus() === OauthClientRequestStatus::Active;

    if ($isActive && $isPendingOrRejected) {
      $consumer = $this->oauthClientConsumerService->createConsumerFromRequest($entity);
      $entity->setClient($consumer);
      $entity->save();
    }
  }

}
