<?php

declare(strict_types=1);

namespace Drupal\oauth_client\Routing;

use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\Routing\DefaultHtmlRouteProvider;
use Drupal\oauth_client\Controller\AccessController;
use Drupal\oauth_client\Form\OauthClientRequestDeleteForm;
use Drupal\oauth_client\Form\OauthClientRequestForm;
use Symfony\Component\Routing\Route;

/**
 * Provides routes for the OAuth Client Request entity.
 */
class OauthClientRequestHtmlRouteProvider extends DefaultHtmlRouteProvider {

  /**
   * {@inheritdoc}
   */
  protected function getAddPageRoute(EntityTypeInterface $entity_type): ?Route {
    if ($route = parent::getAddPageRoute($entity_type)) {
      $route->addRequirements([
        '_custom_access' => AccessController::class . '::accessUserRequests',
        '_entity_create_any_access' => $entity_type->id(),
      ]);
      return $route;
    }
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  protected function getAddFormRoute(EntityTypeInterface $entity_type): ?Route {
    if (!$entity_type->hasLinkTemplate('add-form')) {
      return NULL;
    }

    $route = new Route($entity_type->getLinkTemplate('add-form'));
    $route->addDefaults([
      '_entity_form' => "{$entity_type->id()}.default",
      '_title_callback' => OauthClientRequestForm::class . '::title',
    ]);
    $route->addRequirements([
      '_entity_create_access' => "{$entity_type->id()}:{oauth_client_request_type}",
      '_custom_access' => AccessController::class . '::accessUserRequests',
    ]);
    $route->setOption('_admin_route', TRUE);
    $route->setOption('parameters', [
      'user' => ['type' => 'entity:user'],
      'oauth_client_request_type' => ['type' => 'entity:oauth_client_request_type'],
    ]);

    return $route;
  }

  /**
   * {@inheritdoc}
   */
  protected function getCanonicalRoute(EntityTypeInterface $entity_type): ?Route {
    if (!$entity_type->hasLinkTemplate('canonical')) {
      return NULL;
    }
    $entityTypeId = $entity_type->id();
    $route = new Route($entity_type->getLinkTemplate('canonical'));
    $route->addDefaults([
      '_entity_form' => "$entityTypeId.default",
      '_title_callback' => OauthClientRequestForm::class . '::title',
    ]);
    $route->addRequirements([
      '_entity_access' => "$entityTypeId.update",
      '_custom_access' => AccessController::class . '::accessUserRequests',
    ]);
    $route->setOption('parameters', [
      'user' => ['type' => 'entity:user'],
      $entityTypeId => ['type' => "entity:$entityTypeId"],
    ]);

    return $route;
  }

  /**
   * {@inheritdoc}
   */
  protected function getDeleteFormRoute(EntityTypeInterface $entity_type): ?Route {
    if (!$entity_type->hasLinkTemplate('delete-form')) {
      return NULL;
    }

    $entityTypeId = $entity_type->id();
    $route = new Route($entity_type->getLinkTemplate('delete-form'));
    $route->addDefaults([
      '_entity_form' => "$entityTypeId.delete",
      '_title_callback' => OauthClientRequestDeleteForm::class . '::title',
    ]);
    $route->addRequirements([
      '_entity_access' => "$entityTypeId.delete",
    ]);
    $route->setOption('_admin_route', TRUE);
    $route->setOption('parameters', [
      'user' => ['type' => 'entity:user'],
      $entityTypeId => ['type' => "entity:$entityTypeId"],
    ]);

    return $route;
  }

}
