<?php

declare(strict_types=1);

namespace src\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\oauth_client\Entity\OauthClientRequest;
use Drupal\oauth_client\Entity\OauthClientRequestStatus;
use Drupal\oauth_client\Entity\OauthClientRequestType;
use Drupal\oauth_client\Permission\OauthClientPermissions;
use Drupal\simple_oauth\Entity\Oauth2Scope;
use Drupal\Tests\user\Traits\UserCreationTrait;

/**
 * Kernel test for OAuth Client notifications.
 *
 * @group oauth_client
 */
class NotificationKernelTest extends KernelTestBase {

  use UserCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'consumers',
    'file',
    'image',
    'oauth_client',
    'options',
    'serialization',
    'simple_oauth',
    'system',
    'user',
    'views',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('consumer');
    $this->installEntitySchema('oauth2_scope');
    $this->installEntitySchema('user');
    $this->installEntitySchema('oauth_client_request');
    $this->installConfig(['oauth_client', 'simple_oauth', 'user', 'system']);

    Oauth2Scope::create(['name' => 'scopeN'])->save();
    OauthClientRequestType::create([
      'id' => 'typeN',
      'label' => 'Type N',
      'grant_type' => ['client_credentials' => ['enabled' => TRUE]],
      'scope' => 'scopeN',
    ])->save();

    // Create a requester user who can request this type.
    $this->createUser([
      OauthClientPermissions::getUserPermission('typeN'),
    ], 'requester', FALSE, ['mail' => 'requester@example.com']);

    // Create two managers with the manage permission.
    user_role_grant_permissions('administrator', ['manage oauth clients']);
    $this->createUser(['manage oauth clients'], 'manager1', FALSE, ['mail' => 'manager1@example.com']);
    $this->createUser(['manage oauth clients'], 'manager2', FALSE, ['mail' => 'manager2@example.com']);

    // Start collecting emails via the test mail collector.
    $this->container->get('config.factory')->getEditable('system.mail')->set('interface.default', 'test_mail_collector')->save();
  }

  /**
   * Tests that no notifications are sent for disabled actions.
   */
  public function testNoNotificationsWhenDisabled(): void {
    $requester = user_load_by_name('requester');

    // Create a request by the requester.
    $request = OauthClientRequest::create([
      'type' => 'typeN',
      'label' => 'Test',
      'uid' => $requester,
      'user' => $requester,
    ]);
    $request->save();

    // Assert no emails were sent (requests are disabled by default).
    $captured = $this->container->get('state')->get('system.test_mail_collector') ?? [];
    $this->assertCount(0, $captured, 'No emails should be sent when request notifications are disabled.');

    // Approve the request.
    $request->setStatus(OauthClientRequestStatus::Active)->save();

    // Assert no emails were sent (approvals are disabled by default).
    $captured = $this->container->get('state')->get('system.test_mail_collector') ?? [];
    $this->assertCount(0, $captured, 'No emails should be sent when approval notifications are disabled.');

    // Reject the request.
    $request->setStatus(OauthClientRequestStatus::Rejected)->save();

    // Assert no emails were sent (rejections are disabled by default).
    $captured = $this->container->get('state')->get('system.test_mail_collector') ?? [];
    $this->assertCount(0, $captured, 'No emails should be sent when rejection notifications are disabled.');
  }

  /**
   * Tests notifications on request creation, approval, and rejection.
   */
  public function testNotificationsOnRequestApproveReject(): void {
    // Enable all notifications.
    $config = $this->container->get('config.factory')->getEditable('oauth_client.settings');
    $config->set('notifications.request.enabled', TRUE);
    $config->set('notifications.approve.enabled', TRUE);
    $config->set('notifications.reject.enabled', TRUE);
    $config->save();

    $requester = user_load_by_name('requester');
    $requesterOptions = ['uid' => $requester, 'user' => $requester];

    // Create two requests by the requester.
    $request1 = OauthClientRequest::create([
      'type' => 'typeN',
      'label' => 'First',
      'status' => OauthClientRequestStatus::Pending->value,
    ] + $requesterOptions);
    $request1->save();

    $managers = ['manager1@example.com', 'manager2@example.com'];
    $this->assertNotificationToEmails($managers);
    // Clear captured emails for next assertion.
    $this->container->get('state')->set('system.test_mail_collector', []);

    $request2 = OauthClientRequest::create([
      'type' => 'typeN',
      'label' => 'Second',
      'status' => OauthClientRequestStatus::Pending->value,
    ] + $requesterOptions);
    $request2->save();

    $this->assertNotificationToEmails($managers);
    $this->container->get('state')->set('system.test_mail_collector', []);

    // Approve the first request.
    $request1->setStatus(OauthClientRequestStatus::Active)->save();
    $this->assertNotificationToEmails([$requester->getEmail()]);
    $this->container->get('state')->set('system.test_mail_collector', []);

    // Reject the second request.
    $request2->setStatus(OauthClientRequestStatus::Rejected)->save();
    $this->assertNotificationToEmails([$requester->getEmail()]);
    $this->container->get('state')->set('system.test_mail_collector', []);
  }

  /**
   * Asserts that notification emails were sent to the expected addresses.
   *
   * @param string[] $emails
   *   The expected email addresses.
   */
  protected function assertNotificationToEmails(array $emails): void {
    $captured = $this->container->get('state')->get('system.test_mail_collector') ?? [];
    $sentTo = array_map(fn($m) => $m['to'] ?? '', $captured);
    sort($sentTo);
    sort($emails);
    $this->assertSame($emails, $sentTo, 'Notification emails were sent to the expected addresses.');

    // Clear captured emails after assertion.
    $this->container->get('state')->set('system.test_mail_collector', []);
  }

}
