<?php

namespace Drupal\oci_osfs\Service;

/**
 * Directory operations manager for OCI Object Storage.
 *
 * Handles listing, directory existence checks, and object enumeration.
 */
class OciDirectoryManager {

  public function __construct(
    protected OciObjectStorageFactory $objectStorageFactory,
    protected OciPathResolver $pathResolver,
    protected OciLogger $logger,
  ) {}

  /**
   * Lists objects in a directory.
   *
   * @param string $scheme
   *   The stream scheme (oci, public, private).
   * @param string $directory
   *   The directory path.
   * @param array $options
   *   Options: recursive, limit, prefix.
   *
   * @return array
   *   Array of object information.
   */
  public function listDirectory(string $scheme, string $directory, array $options = []): array {
    try {
      $prefix = $this->pathResolver->toObjectKey($scheme, $directory);
      if ($prefix && !str_ends_with($prefix, '/')) {
        $prefix .= '/';
      }

      $client = $this->objectStorageFactory->createClient();
      $ns = $this->pathResolver->getNamespace();
      $bucket = $this->pathResolver->getBucket();

      // PSEUDOCODE for actual implementation:
      // $params = [
      //   'namespaceName' => $ns,
      //   'bucketName' => $bucket,
      //   'prefix' => $prefix,
      //   'limit' => $options['limit'] ?? 1000,
      // ];
      //
      // if (!empty($options['delimiter']) && !$options['recursive']) {
      //   $params['delimiter'] = '/';
      // }
      //
      // $response = $client->listObjects($params);
      // $objects = [];
      //
      // foreach ($response->getData()->getObjects() as $obj) {
      //   $objects[] = [
      //     'name' => $obj->getName(),
      //     'size' => $obj->getSize(),
      //     'etag' => $obj->getEtag(),
      //     'time_created' => $obj->getTimeCreated(),
      //     'time_modified' => $obj->getTimeModified(),
      //   ];
      // }
      //
      // return $objects;

      // Placeholder implementation until SDK is properly configured.
      $this->logger->logOperation('list_directory', [
        'prefix' => $prefix,
        'scheme' => $scheme,
      ]);

      return [];
    }
    catch (\Throwable $e) {
      $this->logger->logError('Directory listing failed', $e, [
        'directory' => $directory,
        'scheme' => $scheme,
      ]);
      throw new \RuntimeException('Failed to list directory: ' . $e->getMessage(), 0, $e);
    }
  }

  /**
   * Checks if a directory exists (has any objects with the prefix).
   *
   * @param string $scheme
   *   The stream scheme.
   * @param string $directory
   *   The directory path.
   *
   * @return bool
   *   TRUE if directory exists.
   */
  public function directoryExists(string $scheme, string $directory): bool {
    try {
      $objects = $this->listDirectory($scheme, $directory, ['limit' => 1]);
      return !empty($objects);
    }
    catch (\Throwable $e) {
      $this->logger->logError('Directory existence check failed', $e, [
        'directory' => $directory,
      ]);
      return FALSE;
    }
  }

  /**
   * Gets directory size (sum of all objects).
   *
   * @param string $scheme
   *   The stream scheme.
   * @param string $directory
   *   The directory path.
   *
   * @return int
   *   Total size in bytes.
   */
  public function getDirectorySize(string $scheme, string $directory): int {
    try {
      $objects = $this->listDirectory($scheme, $directory, ['recursive' => TRUE]);
      return array_sum(array_column($objects, 'size'));
    }
    catch (\Throwable $e) {
      $this->logger->logError('Directory size calculation failed', $e, [
        'directory' => $directory,
      ]);
      return 0;
    }
  }

  /**
   * Counts objects in a directory.
   *
   * @param string $scheme
   *   The stream scheme.
   * @param string $directory
   *   The directory path.
   * @param bool $recursive
   *   Whether to count recursively.
   *
   * @return int
   *   Number of objects.
   */
  public function countObjects(string $scheme, string $directory, bool $recursive = FALSE): int {
    try {
      $objects = $this->listDirectory($scheme, $directory, ['recursive' => $recursive]);
      return count($objects);
    }
    catch (\Throwable $e) {
      $this->logger->logError('Object counting failed', $e, [
        'directory' => $directory,
      ]);
      return 0;
    }
  }

  /**
   * Deletes all objects in a directory.
   *
   * @param string $scheme
   *   The stream scheme.
   * @param string $directory
   *   The directory path.
   * @param bool $recursive
   *   Whether to delete recursively.
   *
   * @return int
   *   Number of objects deleted.
   */
  public function deleteDirectory(string $scheme, string $directory, bool $recursive = FALSE): int {
    try {
      $objects = $this->listDirectory($scheme, $directory, ['recursive' => $recursive]);
      $deleted = 0;

      $client = $this->objectStorageFactory->createClient();
      $ns = $this->pathResolver->getNamespace();
      $bucket = $this->pathResolver->getBucket();

      foreach ($objects as $object) {
        // PSEUDOCODE:
        // $client->deleteObject([
        //   'namespaceName' => $ns,
        //   'bucketName' => $bucket,
        //   'objectName' => $object['name'],
        // ]);
        $deleted++;
      }

      $this->logger->logOperation('delete_directory', [
        'directory' => $directory,
        'deleted' => $deleted,
      ]);

      return $deleted;
    }
    catch (\Throwable $e) {
      $this->logger->logError('Directory deletion failed', $e, [
        'directory' => $directory,
      ]);
      throw new \RuntimeException('Failed to delete directory: ' . $e->getMessage(), 0, $e);
    }
  }

}
