<?php

namespace Drupal\oci_osfs\Service;

use Drupal\Core\Config\ConfigFactoryInterface;

class OciPathResolver {

  public function __construct(
    protected ConfigFactoryInterface $configFactory
  ) {}

  public function getBucket(): string {
    $c = $this->configFactory->get('oci_osfs.settings');
    $bucket = (string) $c->get('bucket');
    if ($bucket === '') {
      throw new \RuntimeException('Bucket not configured.');
    }
    return $bucket;
  }

  public function getNamespace(): string {
    $c = $this->configFactory->get('oci_osfs.settings');
    $ns = (string) $c->get('namespace');
    if ($ns === '') {
      throw new \RuntimeException('Namespace not configured.');
    }
    return $ns;
  }

  public function toObjectKey(string $scheme, string $target): string {
    $c = $this->configFactory->get('oci_osfs.settings');
    $target = ltrim($target, '/');

    if (str_contains($target, '..')) {
      throw new \InvalidArgumentException('Invalid path traversal.');
    }

    $root = trim((string) $c->get('root_prefix'), '/');
    $public = trim((string) ($c->get('public_prefix') ?: 'public'), '/');
    $private = trim((string) ($c->get('private_prefix') ?: 'private'), '/');

    $base = match ($scheme) {
      'public' => $public,
      'private' => $private,
      default => '',
    };

    $parts = array_filter([$root, $base, $target]);
    $key = implode('/', $parts);

    $this->assertAllowedPrefix($key);

    return $key;
  }

  protected function assertAllowedPrefix(string $key): void {
    $c = $this->configFactory->get('oci_osfs.settings');
    $allowed = (array) $c->get('allowed_prefixes');
    if (!$allowed) {
      return;
    }

    $k = trim($key, '/');
    foreach ($allowed as $p) {
      $p = trim((string) $p, '/');
      if ($p !== '' && (str_starts_with($k, $p . '/') || $k === $p)) {
        return;
      }
    }

    throw new \InvalidArgumentException('Access to this prefix is not allowed.');
  }

}
