<?php

namespace Drupal\oci_osfs\StreamWrapper;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\StreamWrapper\StreamWrapperInterface;
use Drupal\oci_osfs\Service\OciMetadataCache;
use Drupal\oci_osfs\Service\OciObjectStorageFactory;
use Drupal\oci_osfs\Service\OciPathResolver;
use Drupal\oci_osfs\Service\OciPhpStreamBridge;
use Drupal\oci_osfs\Service\OciUrlGenerator;
use Drupal\oci_osfs\Service\S3ClientFactory;
use Aws\S3\Exception\S3Exception;

/**
 * Drupal stream wrapper for OCI Object Storage.
 */
class OciStreamWrapper implements StreamWrapperInterface {

  public $context;
  protected $uri;
  protected $handle;
  protected $mode;
  protected $buffer;
  protected $position;
  protected $objectSize;

  protected ConfigFactoryInterface $configFactory;
  protected OciObjectStorageFactory $objectStorageFactory;
  protected OciPathResolver $pathResolver;
  protected OciUrlGenerator $urlGenerator;
  protected OciMetadataCache $metadataCache;
  protected OciPhpStreamBridge $phpStreamBridge;
  protected S3ClientFactory $s3ClientFactory;
  protected $logger;
  protected $rateLimiter;

  public function __construct(
    ?ConfigFactoryInterface $configFactory = NULL,
    ?OciObjectStorageFactory $objectStorageFactory = NULL,
    ?OciPathResolver $pathResolver = NULL,
    ?OciUrlGenerator $urlGenerator = NULL,
    ?OciMetadataCache $metadataCache = NULL,
    ?OciPhpStreamBridge $phpStreamBridge = NULL,
    $logger = NULL,
    $rateLimiter = NULL,
  ) {
    if ($configFactory === NULL) {
      $this->configFactory = \Drupal::service('config.factory');
      $this->objectStorageFactory = \Drupal::service('oci_osfs.object_storage_factory');
      $this->pathResolver = \Drupal::service('oci_osfs.path_resolver');
      $this->urlGenerator = \Drupal::service('oci_osfs.url_generator');
      $this->metadataCache = \Drupal::service('oci_osfs.metadata_cache');
      $this->phpStreamBridge = \Drupal::service('oci_osfs.php_stream_bridge');
      $this->s3ClientFactory = \Drupal::service('oci_osfs.s3_client_factory');
      $this->logger = \Drupal::service('oci_osfs.logger');
      $this->rateLimiter = \Drupal::service('oci_osfs.rate_limiter');
    }
    else {
      $this->configFactory = $configFactory;
      $this->objectStorageFactory = $objectStorageFactory;
      $this->pathResolver = $pathResolver;
      $this->urlGenerator = $urlGenerator;
      $this->metadataCache = $metadataCache;
      $this->phpStreamBridge = $phpStreamBridge;
      $this->s3ClientFactory = \Drupal::service('oci_osfs.s3_client_factory');
      $this->logger = $logger;
      $this->rateLimiter = $rateLimiter;
    }
  }

  public static function getType() {
    return StreamWrapperInterface::NORMAL;
  }

  public function getName() {
    return t('OCI Object Storage');
  }

  public function getDescription() {
    return t('Official OCI SDK-backed filesystem.');
  }

  public function setUri($uri) {
    $this->uri = $uri;
  }

  public function getUri() {
    return $this->uri;
  }

  protected function getTarget($uri = NULL) {
    if (!isset($uri)) {
      $uri = $this->uri;
    }
    [, $target] = explode('://', $uri, 2);
    return trim($target, '\/');
  }

  public function getExternalUrl() {
    $config = $this->configFactory->get('oci_osfs.settings');
    $scheme = $this->getScheme();
    $target = $this->getTarget();

    // For image style derivatives, return a Drupal-managed URL
    // This allows Drupal's image style controller to generate derivatives on-demand
    if (strpos($target, 'styles/') === 0) {
      // Build the path for Drupal's image style controller
      // Format: /system/files/styles/{style_name}/{scheme}/{path}
      // Example: oci://styles/wide/oci/2025-12/image.png
      //       -> /system/files/styles/wide/oci/2025-12/image.png

      // Extract style name and actual path
      // target = "styles/wide/oci/2025-12/image.png"
      $parts = explode('/', $target, 3);
      if (count($parts) >= 3 && $parts[0] === 'styles') {
        $style_name = $parts[1]; // "wide"
        $rest = $parts[2]; // "oci/2025-12/image.png"

        $base_url = \Drupal::request()->getSchemeAndHttpHost();
        return $base_url . '/system/files/styles/' . $style_name . '/' . $rest;
      }
    }

    if ($scheme === 'public' || ($scheme === 'oci' && $config->get('public_delivery') === 'direct')) {
      return $this->urlGenerator->publicUrl($scheme, $target);
    }
    return NULL;
  }

  public function stream_open($uri, $mode, $options, &$opened_path) {
    $this->setUri($uri);
    $this->mode = $mode;
    $this->buffer = '';
    $this->position = 0;
    $this->objectSize = 0;
    $opened_path = $uri;

    try {
      $config = $this->configFactory->get('oci_osfs.settings');
      $bucket = $config->get('bucket');
      $key = $this->keyFromUri($uri);
      $client = $this->s3ClientFactory->getClient();

      // For read modes, try to get the object
      if (strpos($mode, 'r') !== FALSE || strpos($mode, 'a') !== FALSE) {
        try {
          $result = $client->getObject([
            'Bucket' => $bucket,
            'Key' => $key,
          ]);
          $this->buffer = (string) $result['Body'];
          $this->objectSize = strlen($this->buffer);

          // For append mode, position at end
          if (strpos($mode, 'a') !== FALSE) {
            $this->position = $this->objectSize;
          }
        }
        catch (S3Exception $e) {
          // File doesn't exist - ok for w, a+ modes
          if (strpos($mode, 'r') !== FALSE && strpos($mode, '+') === FALSE) {
            if ($options & STREAM_REPORT_ERRORS) {
              trigger_error("fopen({$uri}): failed to open stream: No such file or object", E_USER_WARNING);
            }
            return FALSE;
          }
        }
      }

      return TRUE;
    }
    catch (\Exception $e) {
      $this->logger->logError('stream_open', $e->getMessage(), ['uri' => $uri]);
      if ($options & STREAM_REPORT_ERRORS) {
        trigger_error("fopen({$uri}): {$e->getMessage()}", E_USER_WARNING);
      }
      return FALSE;
    }
  }

  public function stream_read($count) {
    if ($this->position >= $this->objectSize) {
      return '';
    }

    $data = substr($this->buffer, $this->position, $count);
    $this->position += strlen($data);
    return $data;
  }

  public function stream_write($data) {
    $data_length = strlen($data);

    // Handle different write positions
    if ($this->position < $this->objectSize) {
      // Overwrite existing data
      $this->buffer = substr($this->buffer, 0, $this->position) . $data . substr($this->buffer, $this->position + $data_length);
    }
    else {
      // Append to buffer
      $this->buffer .= $data;
    }

    $this->position += $data_length;
    $this->objectSize = max($this->objectSize, $this->position);

    return $data_length;
  }

  public function stream_tell() {
    return $this->position;
  }

  public function stream_eof() {
    return $this->position >= $this->objectSize;
  }

  public function stream_seek($offset, $whence = SEEK_SET) {
    $new_position = $this->position;

    switch ($whence) {
      case SEEK_SET:
        $new_position = $offset;
        break;
      case SEEK_CUR:
        $new_position = $this->position + $offset;
        break;
      case SEEK_END:
        $new_position = $this->objectSize + $offset;
        break;
      default:
        return FALSE;
    }

    if ($new_position < 0) {
      return FALSE;
    }

    $this->position = $new_position;
    return TRUE;
  }

  public function stream_flush() {
    return $this->writeBufferToS3();
  }

  public function stream_close() {
    // Flush any remaining data
    if ($this->mode && (strpos($this->mode, 'w') !== FALSE || strpos($this->mode, 'a') !== FALSE || strpos($this->mode, '+') !== FALSE)) {
      $this->writeBufferToS3();
    }

    // Clear state
    $this->buffer = '';
    $this->position = 0;
    $this->objectSize = 0;
    $this->mode = NULL;
  }

  protected function writeBufferToS3(): bool {
    if (!$this->uri || !$this->buffer) {
      return TRUE;
    }

    try {
      $config = $this->configFactory->get('oci_osfs.settings');
      $bucket = $config->get('bucket');
      $key = $this->keyFromUri($this->uri);
      $client = $this->s3ClientFactory->getClient();

      $client->putObject([
        'Bucket' => $bucket,
        'Key' => $key,
        'Body' => $this->buffer,
        'ContentType' => $this->getMimeType($key),
      ]);

      // Update metadata cache immediately with fresh data
      $size = strlen($this->buffer);
      $mtime = time();
      $this->metadataCache->set($bucket, $key, [
        'size' => $size,
        'mtime' => $mtime,
      ]);

      $this->logger->logOperation('write', ['key' => $key, 'size' => $size]);

      return TRUE;
    }
    catch (\Exception $e) {
      $this->logger->logError('stream_flush', $e->getMessage(), ['uri' => $this->uri]);
      return FALSE;
    }
  }

  protected function getMimeType(string $key): string {
    $extension = pathinfo($key, PATHINFO_EXTENSION);
    $mime_types = [
      'jpg' => 'image/jpeg',
      'jpeg' => 'image/jpeg',
      'png' => 'image/png',
      'gif' => 'image/gif',
      'webp' => 'image/webp',
      'avif' => 'image/avif',
      'pdf' => 'application/pdf',
      'txt' => 'text/plain',
      'html' => 'text/html',
      'css' => 'text/css',
      'js' => 'application/javascript',
      'json' => 'application/json',
      'xml' => 'application/xml',
      'zip' => 'application/zip',
    ];

    return $mime_types[$extension] ?? 'application/octet-stream';
  }

  public function stream_stat() {
    return $this->buildFileStat($this->objectSize, time());
  }

  public function unlink($uri) {
    try {
      $this->setUri($uri);
      $config = $this->configFactory->get('oci_osfs.settings');
      $bucket = $config->get('bucket');
      $key = $this->keyFromUri($uri);
      $client = $this->s3ClientFactory->getClient();

      $client->deleteObject([
        'Bucket' => $bucket,
        'Key' => $key,
      ]);

      // Clear metadata cache
      $this->metadataCache->delete($bucket, $key);

      $this->logger->logOperation('delete', ['key' => $key]);

      return TRUE;
    }
    catch (\Exception $e) {
      $this->logger->logError('unlink', $e->getMessage(), ['uri' => $uri]);
      return FALSE;
    }
  }

  public function rename($from_uri, $to_uri) {
    try {
      $config = $this->configFactory->get('oci_osfs.settings');
      $bucket = $config->get('bucket');
      $from_key = $this->keyFromUri($from_uri);
      $to_key = $this->keyFromUri($to_uri);
      $client = $this->s3ClientFactory->getClient();

      // Copy object to new location
      $client->copyObject([
        'Bucket' => $bucket,
        'CopySource' => "{$bucket}/{$from_key}",
        'Key' => $to_key,
      ]);

      // Delete original
      $client->deleteObject([
        'Bucket' => $bucket,
        'Key' => $from_key,
      ]);

      // Clear metadata cache for both keys
      $this->metadataCache->delete($bucket, $from_key);
      $this->metadataCache->delete($bucket, $to_key);

      $this->logger->logOperation('rename', ['from' => $from_key, 'to' => $to_key]);

      return TRUE;
    }
    catch (\Exception $e) {
      $this->logger->logError('rename', $e->getMessage(), ['from' => $from_uri, 'to' => $to_uri]);
      return FALSE;
    }
  }

  public function mkdir($uri, $mode, $options) {
    // Object storage doesn't need explicit directories
    // They're created implicitly when objects with paths are uploaded
    return TRUE;
  }

  public function rmdir($uri, $options) {
    // Object storage doesn't have real directories
    // Just return true as there's nothing to remove
    return TRUE;
  }

  public function url_stat($uri, $flags) {
    try {
      $this->setUri($uri);
      $config = $this->configFactory->get('oci_osfs.settings');
      $bucket = $config->get('bucket');
      $key = $this->keyFromUri($uri);

      // Check cache first
      $cached = $this->metadataCache->get($bucket, $key);
      if ($cached !== NULL) {
        return $this->buildFileStat($cached['size'], $cached['mtime']);
      }

      $client = $this->s3ClientFactory->getClient();

      // Try to get object metadata
      try {
        $result = $client->headObject([
          'Bucket' => $bucket,
          'Key' => $key,
        ]);

        $size = (int) $result['ContentLength'];
        $mtime = $result['LastModified'] ? $result['LastModified']->getTimestamp() : time();

        // Cache the metadata
        $this->metadataCache->set($bucket, $key, [
          'size' => $size,
          'mtime' => $mtime,
        ]);

        return $this->buildFileStat($size, $mtime);
      }
      catch (S3Exception $e) {
        // Object doesn't exist
        // If this looks like a directory path (no file extension or ends with /),
        // return a fake directory stat to make is_writable() work
        $target = $this->getTarget();
        $is_directory_path = (substr($target, -1) === '/' || !pathinfo($target, PATHINFO_EXTENSION));

        if (($flags & STREAM_URL_STAT_QUIET) && $is_directory_path) {
          // Return directory stat instead of FALSE
          // This makes is_writable() return TRUE for directory paths
          return $this->buildDirectoryStat();
        }

        // File doesn't exist
        if ($flags & STREAM_URL_STAT_QUIET) {
          return FALSE;
        }
        trigger_error("stat(): No such file or object: {$uri}", E_USER_WARNING);
        return FALSE;
      }
    }
    catch (\Exception $e) {
      $this->logger->logError('url_stat', $e->getMessage(), ['uri' => $uri]);
      if (!($flags & STREAM_URL_STAT_QUIET)) {
        trigger_error("stat(): {$e->getMessage()}", E_USER_WARNING);
      }
      return FALSE;
    }
  }

  protected function buildDirectoryStat(): array {
    $time = time();
    return [
      0 => 0, 1 => 0, 2 => 0040777,  // directory with rwxrwxrwx
      3 => 0, 4 => 0, 5 => 0, 6 => 0,
      7 => 0,  // size 0 for directories
      8 => $time, 9 => $time, 10 => $time,
      11 => -1, 12 => -1,
      'dev' => 0, 'ino' => 0, 'mode' => 0040777,  // directory mode
      'nlink' => 0, 'uid' => 0, 'gid' => 0,
      'rdev' => 0, 'size' => 0,
      'atime' => $time, 'mtime' => $time, 'ctime' => $time,
      'blksize' => -1, 'blocks' => -1,
    ];
  }

  public function dir_opendir($uri, $options) {
    return FALSE;
  }

  public function dir_readdir() {
    return FALSE;
  }

  public function dir_rewinddir() {
    return FALSE;
  }

  public function dir_closedir() {
    return FALSE;
  }

  protected function getScheme() {
    if (!isset($this->uri)) {
      return 'oci';
    }
    $parts = explode('://', $this->uri, 2);
    return $parts[0] ?? 'oci';
  }

  protected function keyFromUri(string $uri): string {
    $this->setUri($uri);
    $target = $this->getTarget();
    return $this->pathResolver->toObjectKey($this->getScheme(), $target);
  }

  protected function buildFileStat(int $size, int $mtime): array {
    return [
      0 => 0, 1 => 0, 2 => 0100666,
      3 => 0, 4 => 0, 5 => 0, 6 => 0,
      7 => $size,
      8 => $mtime, 9 => $mtime, 10 => $mtime,
      11 => -1, 12 => -1,
      'dev' => 0, 'ino' => 0, 'mode' => 0100666,
      'nlink' => 0, 'uid' => 0, 'gid' => 0,
      'rdev' => 0, 'size' => $size,
      'atime' => $mtime, 'mtime' => $mtime, 'ctime' => $mtime,
      'blksize' => -1, 'blocks' => -1,
    ];
  }

  public function stream_lock($operation) {
    return FALSE;
  }

  public function stream_metadata($uri, $option, $value) {
    return FALSE;
  }

  public function stream_truncate($new_size) {
    return FALSE;
  }

  public function stream_set_option($option, $arg1, $arg2) {
    return FALSE;
  }

  public function stream_cast($cast_as) {
    return FALSE;
  }


  public function realpath() {
    return FALSE;
  }

  public function dirname($uri = NULL) {
    [, $target] = explode('://', $uri ?: $this->uri, 2);
    $dirname = dirname($target);
    return $this->getScheme() . '://' . $dirname;
  }


}
