# OCI OSFS Module - Deployment Guide

## Quick Deployment

### 1. Install Module

**All dependencies (including `aws/aws-sdk-php`) are installed automatically!**

```bash
# Install module via Composer (recommended)
composer require drupal/oci_osfs

# Or if installed manually in custom modules directory
cd web/modules/custom/oci_osfs && composer install

# Enable module
drush en oci_osfs -y

# Clear cache
drush cr
```

### 2. Configure Credentials

Choose **ONE** of two configuration methods:

#### Method A: Admin UI (New in 1.0.0-alpha)

**Easiest for most users!**

1. Visit `/admin/config/media/oci-osfs`
2. **Uncheck** "Use credentials from settings.php instead of form"
3. Enter credentials directly in the form:
   - Access Key ID
   - Secret Access Key
4. Configure bucket settings (Region, Namespace, Bucket)
5. Save configuration

**Note:** Credentials are stored in Drupal configuration (encrypted if using config encryption).

#### Method B: settings.php (Recommended for Production)

**For environments requiring external credential management.**

Choose **ONE** authentication method:

##### Option 1: Customer Secret Keys (Recommended)

**No private key files needed!**

**Get from OCI Console:**
Profile → User Settings → Customer Secret Keys → Generate Secret Key

**.env:**

```bash
OCI_ACCESS_KEY="your-access-key"
OCI_SECRET_KEY="your-secret-key"
OCI_OBJECT_NAMESPACE="your-namespace"
OCI_REGION="me-jeddah-1"
OCI_OBJECT_BUCKET="your-bucket"
```

**settings.php:**

```php
// Load .env file (for local development)
$env_file = $app_root . '/../.env';
if (file_exists($env_file)) {
  $env_lines = file($env_file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
  foreach ($env_lines as $line) {
    if (strpos(trim($line), '#') === 0 || strpos($line, '=') === false) {
      continue;
    }
    list($name, $value) = explode('=', $line, 2);
    $name = trim($name);
    $value = trim($value, " \t\n\r\0\x0B\"'");
    if (!getenv($name)) {
      putenv("$name=$value");
    }
  }
}

// Customer Secret Keys
$settings['oci_osfs.customer_keys'] = [
  'access_key' => getenv('OCI_ACCESS_KEY'),
  'secret_key' => getenv('OCI_SECRET_KEY'),
];

// Default bucket settings
$settings['oci_osfs.defaults'] = [
  'region' => getenv('OCI_REGION'),
  'namespace' => getenv('OCI_OBJECT_NAMESPACE'),
  'bucket' => getenv('OCI_OBJECT_BUCKET'),
];

// Allow oci:// scheme for image styles
$settings['file_additional_public_schemes'] = ['oci'];
```

##### Option 2: API Key

**settings.php:**

```php
$settings['oci_osfs.api_key'] = [
  'tenancy_ocid' => getenv('OCI_TENANCY_OCID'),
  'user_ocid' => getenv('OCI_USER_OCID'),
  'fingerprint' => getenv('OCI_FINGERPRINT'),
  'private_key' => getenv('OCI_PRIVATE_KEY'),  // Content (no file path!)
  'passphrase' => getenv('OCI_PRIVATE_KEY_PASSPHRASE'),
];
```

##### Option 3: Instance Principals

**For OCI Compute only - no configuration needed!**

Just ensure IAM policies are configured (see below).

### 3. Configure Module

Visit `/admin/config/media/oci-osfs` (Settings tab) and configure:

- **Credentials:** Choose Admin UI or settings.php method
- **Region:** e.g., me-jeddah-1
- **Namespace:** Your OCI Object Storage namespace
- **Bucket:** Bucket name
- **Prefixes:** Root prefix, public/private prefixes
- **Delivery methods:** Direct URLs or presigned URLs

**Authentication is auto-detected** - Settings page shows which method is active!

### 4. Validate & Test (New in 1.0.0-alpha)

Visit `/admin/config/media/oci-osfs/actions` (Actions tab) and use:

1. **Validate Configuration** - Tests OCI connection, creates test object, verifies access
2. **Refresh Metadata Cache** - Clears cached file information
3. **Copy Local Files to OCI** - Migrates existing files to Object Storage

**Or test via CLI:**

```bash
# Test authentication
drush php:eval "\$auth = \Drupal::service('oci_osfs.auth_factory')->createAuth(); print_r(\$auth);"

# Test file upload
drush php:eval "file_put_contents('oci://test.txt', 'Hello OCI!');"

# Test file read
drush php:eval "echo file_get_contents('oci://test.txt');"
```

---

## Environment-Specific Setup

### Development

```bash
# .env
OCI_ACCESS_KEY="dev-key"
OCI_SECRET_KEY="dev-secret"
OCI_OBJECT_BUCKET="dev-bucket"
```

### Staging

```bash
# .env
OCI_ACCESS_KEY="staging-key"
OCI_SECRET_KEY="staging-secret"
OCI_OBJECT_BUCKET="staging-bucket"
```

### Production

```bash
# .env (or use secrets manager)
OCI_ACCESS_KEY="prod-key"
OCI_SECRET_KEY="prod-secret"
OCI_OBJECT_BUCKET="prod-bucket"
```

**Always use different credentials per environment!**

---

## Instance Principals (OCI Compute)

### Dynamic Group

```text
Any {instance.compartment.id = 'ocid1.compartment.oc1..xxx'}
```

### IAM Policy

```text
Allow dynamic-group <group-name> to manage buckets in compartment <compartment>
Allow dynamic-group <group-name> to manage objects in compartment <compartment>
Allow dynamic-group <group-name> to manage objectstorage-namespaces in compartment <compartment>
```

---

## Security Checklist

- [ ] `.env` added to `.gitignore`
- [ ] Different credentials for dev/staging/prod
- [ ] Environment variables (never hardcode)
- [ ] SSL/TLS enabled
- [ ] Regular credential rotation
- [ ] Monitor logs: `drush watchdog:show --type=oci_osfs`
- [ ] Rate limits configured
- [ ] Allowed prefixes set (if needed)

---

## Monitoring & Logs

### View Logs

```bash
# All OCI logs
drush watchdog:show --type=oci_osfs

# Errors only
drush watchdog:show --type=oci_osfs --severity=Error

# Real-time tail
drush watchdog:tail --type=oci_osfs
```

### Check Status

```bash
# Module info
drush pm:info oci_osfs

# Config
drush config:get oci_osfs.settings

# Requirements
drush status-report | grep -i oci
```

---

## Troubleshooting

### Authentication Failed

```bash
# Check which method is active
drush php:eval "\$auth = \Drupal::service('oci_osfs.auth_factory')->createAuth(); echo \$auth['type'];"

# Check credentials loaded
drush php:eval "echo getenv('OCI_ACCESS_KEY') ? 'SET' : 'NOT SET';"
```

### Environment Variables Not Loading

```bash
# Restart web server/DDEV
ddev restart

# Verify .env location
ls -la .env
```

### Clear Cache

```bash
drush cr
```

### Stream Wrapper Not Found

```bash
# Check registered wrappers
drush php:eval "print_r(stream_get_wrappers());" | grep oci

# Rebuild
drush cr
```

---

## Performance Optimization

### Enable Metadata Caching

```php
// settings.php
$config['oci_osfs.settings']['metadata_cache_max_age'] = 300;  // 5 minutes
```

### Use Direct URLs for Public Files

```php
$config['oci_osfs.settings']['public_delivery'] = 'direct';
```

### Configure CDN (Optional)

Point CloudFront/CDN to your OCI bucket for better performance.

---

## Backup & Disaster Recovery

### Backup Strategy

1. **Database**: Regular Drupal database backups
2. **Files**: OCI Object Storage is already redundant
3. **Configuration**: Export config: `drush config:export`

### Recovery

```bash
# Restore database
drush sql:cli < backup.sql

# Import config
drush config:import

# Clear cache
drush cr
```

---

## Updates & Maintenance

### Update Module

```bash
# Pull latest changes
git pull

# Update dependencies
composer update

# Run updates
drush updatedb

# Clear cache
drush cr
```

### Rotate Credentials

**Customer Secret Keys:**

1. Generate new key in OCI Console
2. Update `.env` file
3. Test: `drush php:eval "file_put_contents('oci://test.txt', 'test');"`
4. Delete old key from OCI Console

**API Key:**

1. Generate new API key
2. Update `.env` with new fingerprint and private key
3. Test authentication
4. Delete old API key

---

## Production Deployment Checklist

- [ ] Module installed: `composer require drupal/oci_osfs` (AWS SDK auto-installed)
- [ ] Module enabled: `drush en oci_osfs -y`
- [ ] Credentials configured (Admin UI or settings.php)
- [ ] Environment variables set (if using settings.php method)
- [ ] `.env` in `.gitignore`
- [ ] Different credentials than dev/staging
- [ ] SSL/TLS enabled
- [ ] Firewall configured
- [ ] Monitoring set up
- [ ] Logs configured
- [ ] Backup strategy in place
- [ ] **Validated configuration using Actions page**
- [ ] **Tested file migration (if migrating existing files)**
- [ ] Tested file upload/download
- [ ] Tested authentication
- [ ] Image styles working (test with `/oci/[style]/[file]` URLs)
- [ ] `file_additional_public_schemes = ['oci']` set in settings.php
- [ ] Documentation updated

---

## File Migration

### Using Actions Page (Recommended)

1. Visit `/admin/config/media/oci-osfs/actions`
2. Navigate to "Copy Local Files to OCI" section
3. Select source scheme (e.g., `public://`, `private://`)
4. Check "Skip files that already exist in OCI"
5. Click "Copy Files"
6. Monitor batch progress

### Using Drush

```bash
# Custom migration script
drush php:eval "
\$file_storage = \Drupal::entityTypeManager()->getStorage('file');
\$files = \$file_storage->loadMultiple();
foreach (\$files as \$file) {
  if (strpos(\$file->getFileUri(), 'public://') === 0) {
    \$old_uri = \$file->getFileUri();
    \$new_uri = str_replace('public://', 'oci://', \$old_uri);
    \$data = file_get_contents(\$old_uri);
    file_put_contents(\$new_uri, \$data);
    \$file->setFileUri(\$new_uri);
    \$file->save();
    echo \"Migrated: \$old_uri -> \$new_uri\\n\";
  }
}
"
```

**Note:** The Actions page provides a better UI with progress tracking and error handling.

---

## Getting Help

### Check Documentation

- [README.md](README.md) - Comprehensive module documentation with all features, setup, and troubleshooting

### Use Actions Page

The Actions page (`/admin/config/media/oci-osfs/actions`) provides:
- **Validate Configuration** - Diagnose connection issues
- **Refresh Metadata Cache** - Fix stale file information
- **Copy Local Files** - Migrate existing files

### Debug

```bash
# Enable verbose logging
drush watchdog:tail --type=oci_osfs

# Test connection using Actions page validation
# Visit: /admin/config/media/oci-osfs/actions

# Or test via CLI:
drush php:eval "
\$factory = \Drupal::service('oci_osfs.object_storage_factory');
try {
  \$client = \$factory->createClient();
  echo 'Connected successfully';
} catch (\Exception \$e) {
  echo 'Error: ' . \$e->getMessage();
}
"
```

---

**Last Updated:** 2025-12-16
**Version:** 1.0.0
**Status:** Ready for Testing
**New Features:** Admin UI credential management, Actions page, File migration tools
**Authentication:** Auto-detected (Customer Keys, API Key, or Instance Principals)
