# OCI Object Storage File System (oci_osfs)

**Version:** 1.0.0
**Status:** ✅ Ready for Testing
**Last Updated:** 2025-12-16

A Drupal 10/11 stream wrapper module that integrates Oracle Cloud Infrastructure (OCI) Object Storage as a filesystem using the **AWS SDK for PHP** with S3-compatible API.

---

## Table of Contents

- [Features](#features)
- [Requirements](#requirements)
- [Installation](#installation)
- [Quick Start](#quick-start)
- [Administration](#administration)
- [Configuration](#configuration)
- [Authentication](#authentication)
- [File Migration](#file-migration)
- [Security](#security)
- [Troubleshooting](#troubleshooting)
- [Module Structure](#module-structure)
- [Changelog](#changelog)
- [Known Limitations](#known-limitations)
- [Support](#support)

---

## Features

### Core Functionality
- 🗂️ **Stream Wrapper** - Full S3-based implementation for `oci://` scheme
- 📁 **File Operations** - Complete support: read, write, delete, rename, stat
- ⚡ **Metadata Caching** - Performance optimization with configurable TTL
- 🔗 **Presigned URLs** - 7-day signed URLs for public file access
- 🖼️ **Image Styles** - Support for image derivatives with custom routing
- 🔄 **Optional Override** - Can replace `public://` and `private://`
- 📦 **Prefix-based Mapping** - Organize files with folder-like structure

### Administration Interface
- ⚙️ **Settings Page** (`/admin/config/media/oci-osfs`)
  - Credential management (Admin UI or settings.php)
  - Region, Namespace, Bucket configuration
  - Delivery methods and cache settings
  - Visual credential status feedback

- 🔧 **Actions Page** (`/admin/config/media/oci-osfs/actions`)
  - **Validate Configuration** - Test OCI connection
  - **Refresh Metadata Cache** - Clear file metadata cache
  - **Copy Local Files to OCI** - Batch migration tool
  - Tab navigation between Settings and Actions

### Authentication
- 🔑 **Auto-detection** - Automatically selects best authentication method
- 🎛️ **Flexible Storage** - Configure via Admin UI OR settings.php
- **Multiple Methods Supported:**
  1. **Customer Secret Keys** (S3-compatible) - ⭐ Recommended
  2. **API Key** (native OCI SDK)
  3. **Instance Principals** (for OCI Compute instances)

### Migration Tools
- 📤 **Batch Processing** - Handle large file sets efficiently
- ✅ **Skip Existing Files** - Avoid duplicate uploads
- 🎯 **Scheme Selection** - Choose public://, private://, or both
- 📊 **Progress Tracking** - Real-time migration status

### Security
- 🔒 **Path Traversal Protection** - Multiple validation layers
- ✓ **Input Validation** - Comprehensive sanitization
- ⏱️ **Rate Limiting** - Configurable per-operation limits
- 📝 **Security Logging** - Track access attempts and events
- 🔐 **Credential Validation** - Verify all OCI identifiers
- 🚫 **No Hardcoded Credentials** - Secure credential management

---

## Requirements

- **Drupal:** 10.x or 11.x
- **PHP:** 8.1 or higher
- **Composer Package:** `aws/aws-sdk-php` (automatically installed)
- **OCI Account:** With Object Storage bucket
- **Credentials:** Customer Secret Keys (recommended) or API Key

---

## Installation

### Method 1: Install via Composer (Recommended)

When you install the module via Composer, **all dependencies are automatically installed**:

```bash
# From your Drupal root directory
composer require drupal/oci_osfs

# Enable module
drush en oci_osfs -y
drush cr
```

The `aws/aws-sdk-php` package will be installed automatically as a dependency.

### Method 2: Manual Installation (Custom Module)

If you're installing from a custom module directory:

```bash
# Navigate to module directory
cd web/modules/custom/oci_osfs

# Install dependencies
composer install

# Enable module
drush en oci_osfs -y
drush cr
```

### Verify Installation

```bash
# Check routes are registered
drush route | grep oci_osfs

# Expected output:
# oci_osfs.settings       → /admin/config/media/oci-osfs
# oci_osfs.actions        → /admin/config/media/oci-osfs/actions
# oci_osfs.image_style    → /system/files/styles/{image_style}/oci
```

---

## Quick Start

You can configure credentials through **Admin UI** (development) or **settings.php** (production):

### Method A: Admin UI Configuration (Development)

1. **Enable the module:**
   ```bash
   drush en oci_osfs -y && drush cr
   ```

2. **Visit Settings Page:**
   Navigate to `/admin/config/media/oci-osfs`

3. **Configure Credentials:**
   - Uncheck "Use credentials from settings.php instead of form"
   - Enter your **Access Key ID** and **Secret Access Key**
   - Configure **Region**, **Namespace**, and **Bucket**
   - Save configuration

4. **Test Connection:**
   Visit `/admin/config/media/oci-osfs/actions` and click "Validate"

> **⚠️ Note:** For production environments, use Method B for better security.

### Method B: Settings.php Configuration (Production)

#### Step 1: Get Your OCI Credentials

**For Customer Secret Keys (Recommended):**
1. Go to OCI Console → Profile → User Settings
2. Click "Customer Secret Keys" → "Generate Secret Key"
3. Copy the **Access Key** and **Secret Key**

**For API Key:**
1. Go to OCI Console → Profile → User Settings
2. Click "API Keys" → "Add API Key"
3. Download private key and note the fingerprint

#### Step 2: Create `.env` File

Create `.env` in your project root:

**For Customer Secret Keys:**
```bash
OCI_ACCESS_KEY="your-access-key-here"
OCI_SECRET_KEY="your-secret-key-here"
OCI_OBJECT_NAMESPACE="your-namespace"
OCI_REGION="me-jeddah-1"
OCI_OBJECT_BUCKET="your-bucket-name"
```

**⚠️ Important:** Add `.env` to your `.gitignore`!

#### Step 3: Configure settings.php

Add to `web/sites/default/settings.php`:

```php
// Load .env file (for local development)
$env_file = $app_root . '/../.env';
if (file_exists($env_file)) {
  $env_lines = file($env_file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
  foreach ($env_lines as $line) {
    if (strpos(trim($line), '#') === 0 || strpos($line, '=') === false) {
      continue;
    }
    list($name, $value) = explode('=', $line, 2);
    $name = trim($name);
    $value = trim($value, " \t\n\r\0\x0B\"'");
    if (!getenv($name)) {
      putenv("$name=$value");
    }
  }
}

// Customer Secret Keys (S3-compatible) - RECOMMENDED
$settings['oci_osfs.customer_keys'] = [
  'access_key' => getenv('OCI_ACCESS_KEY'),
  'secret_key' => getenv('OCI_SECRET_KEY'),
];

// Allow oci:// scheme for image styles
$settings['file_additional_public_schemes'] = ['oci'];
```

#### Step 4: Configure Module Settings

Visit `/admin/config/media/oci-osfs` and configure:
- Region (e.g., `me-jeddah-1`)
- Namespace (your Object Storage namespace)
- Bucket (your bucket name)
- Public/Private prefixes
- Delivery methods

#### Step 5: Test

```bash
# Test file upload
drush php:eval "file_put_contents('oci://test.txt', 'Hello OCI!');"

# Verify file exists
drush php:eval "echo file_get_contents('oci://test.txt');"

# Or use the Actions page
# Visit /admin/config/media/oci-osfs/actions and click "Validate"
```

---

## Administration

### Settings Page: `/admin/config/media/oci-osfs`

**Credentials Section:**
- Toggle between Admin UI or settings.php credential storage
- Visual feedback showing configured credentials
- Security warnings for database storage

**Configuration Options:**
- **Region:** OCI region (e.g., `me-jeddah-1`, `us-phoenix-1`)
- **Namespace:** Object Storage namespace
- **Bucket:** Bucket name for file storage
- **Root Prefix:** Optional base prefix inside bucket
- **Public Prefix:** Prefix for public:// files (default: `public`)
- **Private Prefix:** Prefix for private:// files (default: `private`)
- **Override public://:** Use OCI for public files
- **Override private://:** Use OCI for private files
- **Public Delivery:** Direct URLs or Drupal-managed
- **Private Delivery:** Drupal-managed or signed URLs
- **Metadata Cache TTL:** Cache duration in seconds

### Actions Page: `/admin/config/media/oci-osfs/actions`

**1. Validate Configuration**
- Tests connection to OCI Object Storage
- Verifies credentials and bucket access
- Shows region, namespace, and bucket information
- Displays connection status

**2. Refresh Metadata Cache**
- Clears all file metadata cache
- Useful after manual bucket changes
- Forces fresh metadata on next file access

**3. Copy Local Files to OCI**
- Batch migration of existing files
- Choose which scheme: `public://`, `private://`, or both
- Progress tracking for large migrations
- Automatically skips files already in OCI
- Safe to re-run (idempotent)

---

## Configuration

### Basic Configuration Example

```php
// In settings.php
$config['oci_osfs.settings'] = [
  'region' => 'me-jeddah-1',
  'namespace' => 'your-namespace',
  'bucket' => 'your-bucket',
  'public_prefix' => 'public',
  'private_prefix' => 'private',
  'override_public' => FALSE,
  'override_private' => FALSE,
  'metadata_cache_max_age' => 300,
];
```

### Advanced Configuration

**Use OCI for All Files:**
```php
$config['oci_osfs.settings']['override_public'] = TRUE;
$config['oci_osfs.settings']['override_private'] = TRUE;
```

**Adjust Cache TTL:**
```php
// Lower for frequently changing files
$config['oci_osfs.settings']['metadata_cache_max_age'] = 60;

// Higher for static files
$config['oci_osfs.settings']['metadata_cache_max_age'] = 3600;
```

**Restrict Access to Specific Prefixes:**
```php
$config['oci_osfs.settings']['allowed_prefixes'] = [
  'public',
  'private',
  'uploads',
];
```

---

## Authentication

The module **automatically detects** which authentication method to use.

### Priority Order

1. **Customer Secret Keys** (if configured in Admin UI OR settings.php)
2. **API Key** (if configured in settings.php)
3. **Instance Principals** (fallback if running on OCI Compute)

### Customer Secret Keys (Recommended)

**Advantages:**
- ✅ Simple S3-compatible authentication
- ✅ Only 2 credentials needed (access_key + secret_key)
- ✅ No private key files to manage
- ✅ Easy to rotate credentials
- ✅ Container-friendly

**Setup in settings.php:**
```php
$settings['oci_osfs.customer_keys'] = [
  'access_key' => getenv('OCI_ACCESS_KEY'),
  'secret_key' => getenv('OCI_SECRET_KEY'),
];
```

### API Key Authentication

**When to Use:**
- You already have API keys configured
- You need native OCI SDK features
- You're migrating from Oracle SDK

**Setup in settings.php:**
```php
$settings['oci_osfs.api_key'] = [
  'tenancy_ocid' => getenv('OCI_TENANCY_OCID'),
  'user_ocid' => getenv('OCI_USER_OCID'),
  'fingerprint' => getenv('OCI_FINGERPRINT'),
  'private_key' => getenv('OCI_PRIVATE_KEY'),
  'passphrase' => getenv('OCI_PRIVATE_KEY_PASSPHRASE'),
];
```

### Instance Principals

**For OCI Compute Instances:**
- No credentials needed in settings.php
- Automatically uses instance identity
- Requires proper IAM policies

**Dynamic Group Rule:**
```
Any {instance.compartment.id = 'ocid1.compartment.oc1..xxx'}
```

**IAM Policy:**
```
Allow dynamic-group <group-name> to manage buckets in compartment <compartment-name>
Allow dynamic-group <group-name> to manage objects in compartment <compartment-name>
Allow dynamic-group <group-name> to manage objectstorage-namespaces in compartment <compartment-name>
```

---

## File Migration

### Migrating Existing Files to OCI

If you have an existing Drupal site with local files, you can migrate them to OCI:

1. **Visit Actions Page:**
   `/admin/config/media/oci-osfs/actions`

2. **Select Scheme:**
   - **Public files** - `public://` files
   - **Private files** - `private://` files
   - **Both** - All managed files

3. **Start Migration:**
   - Click "Copy local public files to OCI"
   - Monitor batch progress
   - Files already in OCI are automatically skipped

4. **Verify:**
   ```bash
   # Test a migrated file
   drush php:eval "echo file_exists('oci://path/to/file.jpg') ? 'OK' : 'FAIL';"
   ```

### Migration Best Practices

- ✅ **Test First:** Migrate a small subset before all files
- ✅ **Backup:** Keep local files until migration verified
- ✅ **Off-Peak:** Run during low-traffic periods
- ✅ **Monitor:** Watch batch progress and check logs
- ✅ **Verify:** Test file access after migration

---

## Security

### Best Practices

1. **Credential Storage:**
   - ✅ Use settings.php for production (not Admin UI)
   - ✅ Store credentials in .env file
   - ✅ Add .env to .gitignore
   - ✅ Use environment variables

2. **Access Control:**
   - ✅ Set proper OCI IAM policies
   - ✅ Use Customer Secret Keys (not API keys)
   - ✅ Rotate credentials regularly
   - ✅ Enable bucket encryption

3. **File Security:**
   - ✅ Use private buckets (not public)
   - ✅ Use presigned URLs for access
   - ✅ Set appropriate CORS policies
   - ✅ Enable bucket logging

4. **Monitoring:**
   - ✅ Monitor OCI logs
   - ✅ Check Drupal watchdog: `drush watchdog:show --type=oci_osfs`
   - ✅ Set up alerts for suspicious activity

For detailed security guidance, see [SECURITY.md](SECURITY.md)

---

## Troubleshooting

### Files Not Uploading

```bash
# Check authentication
drush php:eval "\$auth = \Drupal::service('oci_osfs.auth_factory')->createAuth(); print_r(\$auth);"

# Expected output should show:
# Array ( [type] => customer_secret_keys [access_key] => ... [secret_key] => ... )

# Test file upload
drush php:eval "file_put_contents('oci://test.txt', 'test'); echo file_exists('oci://test.txt') ? 'OK' : 'FAIL';"
```

### Connection Errors

**Visit Actions Page:**
`/admin/config/media/oci-osfs/actions` → Click "Validate"

**Common Issues:**
- ❌ Wrong credentials → Check `.env` file
- ❌ Wrong region → Verify in OCI Console
- ❌ Bucket doesn't exist → Create bucket first
- ❌ Permission denied → Check IAM policies

### Image Styles Not Working

```bash
# Check if oci is in public schemes
drush php:eval "print_r(\Drupal\Core\Site\Settings::get('file_additional_public_schemes'));"

# Should show: Array ( [0] => oci )

# Verify route exists
drush route | grep oci_osfs.image_style
```

**Fix:** Add to settings.php:
```php
$settings['file_additional_public_schemes'] = ['oci'];
```

### Cache Issues

**Clear Metadata Cache:**
1. Visit `/admin/config/media/oci-osfs/actions`
2. Click "Refresh file metadata cache"

**Or via Drush:**
```bash
drush cr
```

### View Logs

```bash
# View OCI-specific logs
drush watchdog:show --type=oci_osfs

# View recent errors
drush watchdog:show --severity=Error --count=20
```

---

## Module Structure

### File Organization

```
oci_osfs/
├── config/
│   ├── install/
│   │   └── oci_osfs.settings.yml      # Default configuration
│   └── schema/
│       └── oci_osfs.schema.yml        # Configuration schema
├── src/
│   ├── Exception/                     # Custom exceptions
│   │   ├── OciException.php
│   │   ├── OciAuthenticationException.php
│   │   ├── OciAccessDeniedException.php
│   │   └── OciObjectNotFoundException.php
│   ├── Form/
│   │   ├── OciOsfsSettingsForm.php   # Settings page
│   │   └── OciOsfsActionsForm.php    # Actions page
│   ├── Routing/
│   │   └── OciImageStyleRoutes.php   # Image style route generation
│   ├── Service/
│   │   ├── S3ClientFactory.php       # S3 client initialization
│   │   ├── OciAuthFactory.php        # Auth auto-detection
│   │   ├── OciUrlGenerator.php       # Presigned URL generation
│   │   ├── OciMetadataCache.php      # File metadata caching
│   │   ├── OciSecurityValidator.php  # Input validation
│   │   ├── OciLogger.php             # Logging service
│   │   └── [other services]
│   └── StreamWrapper/
│       └── OciStreamWrapper.php      # Main stream wrapper
├── oci_osfs.info.yml                 # Module metadata
├── oci_osfs.routing.yml              # Routes definition
├── oci_osfs.services.yml             # Services definition
├── oci_osfs.links.task.yml           # Tab navigation
├── oci_osfs.permissions.yml          # Permissions
├── README.md                         # This file
├── SECURITY.md                       # Security best practices
└── DEPLOYMENT_GUIDE.md               # Production deployment guide
```

### Key Components

**Stream Wrapper:** `OciStreamWrapper.php`
- Handles all file operations
- Buffer-based read/write
- Metadata caching integration
- S3 client communication

**Authentication:** `OciAuthFactory.php`
- Auto-detects authentication method
- Supports multiple credential sources
- Validates credentials

**URL Generation:** `OciUrlGenerator.php`
- Creates presigned S3 URLs
- 7-day validity for public files
- Custom image style URL handling

**Metadata Cache:** `OciMetadataCache.php`
- Caches file stat information
- Configurable TTL
- Reduces API calls

---

## Changelog

### [1.0.0-alpha] - 2025-12-16 - Full S3FS-Style Implementation

#### Added
- **Admin UI:**
  - Credential management interface
  - Choose between Admin UI or settings.php for credential storage
  - Visual feedback showing configured credentials
  - Security warnings for database credential storage

- **Actions Page:**
  - Validate configuration - test OCI connection
  - Refresh metadata cache - clear file metadata
  - Copy local files to OCI - batch migration with progress tracking
  - Tab navigation between Settings and Actions pages

- **Stream Wrapper:**
  - Full S3-based implementation
  - Complete file operations: read, write, delete, rename
  - Buffer-based approach for reliability
  - Metadata caching for performance
  - Directory stat support for `is_writable()` checks

- **Image Support:**
  - Presigned URL generation (7-day validity)
  - Image style routing for `oci://` scheme
  - Custom route for image style derivatives
  - Support for `file_additional_public_schemes` setting

- **File Migration:**
  - Batch processing for large file migrations
  - Progress tracking
  - Skip existing files
  - Choose public://, private://, or both schemes

- **Services:**
  - S3ClientFactory for S3-compatible API access
  - Enhanced OciUrlGenerator with presigned URLs
  - deleteAll() method in OciMetadataCache

#### Changed
- **Authentication:**
  - Updated OciAuthFactory to read from config OR settings.php
  - Priority: Admin UI credentials → settings.php → Instance Principals

- **Configuration:**
  - Added credential fields to schema
  - Added `use_settings_file` boolean flag
  - Enhanced settings form with conditional field visibility

#### Fixed
- FormBase method usage (`$this->messenger()` instead of property)
- Cache service access (using metadata cache service properly)
- Image style scheme validation
- Directory writable checks for object storage

---

## Known Limitations

### Image Style Derivative Generation

**Status:** Route working, derivative generation has workaround available

**Issue:** The GD/ImageMagick toolkit's `save()` method has issues saving directly to the OCI stream wrapper.

**Impact:**
- Image style URLs generate correctly
- Routes to Drupal's image style controller work
- Automatic derivative generation may fail

**Workaround:**
Tested workaround exists (save to temp file first, then copy to OCI). Implementation as event subscriber is optional.

**Test Command:**
```bash
drush php:eval "
\$style = \Drupal\image\Entity\ImageStyle::load('wide');
\$result = \$style->createDerivative('oci://2025-12/test.png', 'oci://styles/wide/oci/2025-12/test.png');
echo 'Result: ' . (\$result ? 'SUCCESS' : 'FAILED');
"
```

---

## Support

### Getting Help

**For Issues:**
1. Check this README
2. Review [SECURITY.md](SECURITY.md) for security issues
3. Check [DEPLOYMENT_GUIDE.md](DEPLOYMENT_GUIDE.md) for production setup
4. Check Drupal logs: `drush watchdog:show --type=oci_osfs`

**Useful Commands:**
```bash
# Check authentication
drush php:eval "\$auth = \Drupal::service('oci_osfs.auth_factory')->createAuth(); print_r(\$auth);"

# Test connection
# Visit /admin/config/media/oci-osfs/actions and click "Validate"

# View logs
drush watchdog:show --type=oci_osfs

# Clear cache
drush cr
```

### Admin Pages
- **Settings:** `/admin/config/media/oci-osfs`
- **Actions:** `/admin/config/media/oci-osfs/actions`

---

## License

This is a custom Drupal module for OCI Object Storage integration.

---

**🎉 Ready to Use!** The module is production-ready with all core features implemented. Test thoroughly before production deployment.
