<?php

namespace Drupal\oci_osfs\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\file\Entity\File;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides actions for OCI Object Storage.
 */
class OciOsfsActionsForm extends FormBase {

  /**
   * The file system service.
   */
  protected FileSystemInterface $fileSystem;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = new static();
    $instance->fileSystem = $container->get('file_system');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'oci_osfs_actions_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['#tree'] = TRUE;

    // Validation section
    $form['validation'] = [
      '#type' => 'details',
      '#title' => $this->t('Validate configuration'),
      '#open' => TRUE,
    ];

    $form['validation']['validate'] = [
      '#type' => 'submit',
      '#value' => $this->t('Validate'),
      '#submit' => ['::validateConfiguration'],
    ];

    $form['validation']['description'] = [
      '#markup' => '<p>' . $this->t('Test the connection to OCI Object Storage using current configuration.') . '</p>',
    ];

    // File metadata cache section
    $form['metadata_cache'] = [
      '#type' => 'details',
      '#title' => $this->t('File Metadata Cache'),
      '#open' => TRUE,
    ];

    $form['metadata_cache']['refresh'] = [
      '#type' => 'submit',
      '#value' => $this->t('Refresh file metadata cache'),
      '#submit' => ['::refreshMetadataCache'],
    ];

    $form['metadata_cache']['description'] = [
      '#markup' => '<p>' . $this->t('Clear the metadata cache for all files in OCI Object Storage.') . '</p>',
    ];

    // Copy local files section
    $form['copy_local'] = [
      '#type' => 'details',
      '#title' => $this->t('Copy Local Files to OCI'),
      '#open' => TRUE,
    ];

    $form['copy_local']['scheme'] = [
      '#type' => 'radios',
      '#title' => $this->t('Which files to copy'),
      '#options' => [
        'public' => $this->t('Public files (public://)'),
        'private' => $this->t('Private files (private://)'),
        'both' => $this->t('Both public and private files'),
      ],
      '#default_value' => 'public',
    ];

    $form['copy_local']['copy_public'] = [
      '#type' => 'submit',
      '#value' => $this->t('Copy local public files to OCI'),
      '#submit' => ['::copyLocalFiles'],
      '#states' => [
        'visible' => [
          ':input[name="copy_local[scheme]"]' => ['value' => 'public'],
        ],
      ],
    ];

    $form['copy_local']['copy_private'] = [
      '#type' => 'submit',
      '#value' => $this->t('Copy local private files to OCI'),
      '#submit' => ['::copyLocalFiles'],
      '#states' => [
        'visible' => [
          ':input[name="copy_local[scheme]"]' => ['value' => 'private'],
        ],
      ],
    ];

    $form['copy_local']['copy_both'] = [
      '#type' => 'submit',
      '#value' => $this->t('Copy all local files to OCI'),
      '#submit' => ['::copyLocalFiles'],
      '#states' => [
        'visible' => [
          ':input[name="copy_local[scheme]"]' => ['value' => 'both'],
        ],
      ],
    ];

    $form['copy_local']['warning'] = [
      '#markup' => '<div class="messages messages--warning">' .
        $this->t('<strong>Important:</strong> This will copy files from your local filesystem to OCI Object Storage. Make sure you have configured OCI credentials before running this action.') .
        '</div>',
    ];

    return $form;
  }

  /**
   * Validate configuration submit handler.
   */
  public function validateConfiguration(array &$form, FormStateInterface $form_state) {
    try {
      $config = $this->config('oci_osfs.settings');
      $bucket = $config->get('bucket');
      $region = $config->get('region');
      $namespace = $config->get('namespace');

      if (empty($bucket) || empty($region) || empty($namespace)) {
        $this->messenger()->addError($this->t('OCI configuration is incomplete. Please configure bucket, region, and namespace.'));
        return;
      }

      // Try to get S3 client and list bucket
      $client = \Drupal::service('oci_osfs.s3_client_factory')->getClient();

      $result = $client->listObjectsV2([
        'Bucket' => $bucket,
        'MaxKeys' => 1,
      ]);

      $this->messenger()->addStatus($this->t('Successfully connected to OCI Object Storage!'));
      $this->messenger()->addStatus($this->t('Bucket: @bucket', ['@bucket' => $bucket]));
      $this->messenger()->addStatus($this->t('Region: @region', ['@region' => $region]));
      $this->messenger()->addStatus($this->t('Namespace: @namespace', ['@namespace' => $namespace]));
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('Failed to connect to OCI Object Storage: @error', [
        '@error' => $e->getMessage(),
      ]));
    }
  }

  /**
   * Refresh metadata cache submit handler.
   */
  public function refreshMetadataCache(array &$form, FormStateInterface $form_state) {
    try {
      // Clear the metadata cache using the service
      $metadata_cache = \Drupal::service('oci_osfs.metadata_cache');
      $metadata_cache->deleteAll();

      $this->messenger()->addStatus($this->t('Metadata cache has been cleared.'));
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('Failed to clear metadata cache: @error', [
        '@error' => $e->getMessage(),
      ]));
    }
  }

  /**
   * Copy local files submit handler.
   */
  public function copyLocalFiles(array &$form, FormStateInterface $form_state) {
    $scheme = $form_state->getValue(['copy_local', 'scheme']);
    $schemes_to_copy = [];

    if ($scheme === 'public' || $scheme === 'both') {
      $schemes_to_copy[] = 'public';
    }
    if ($scheme === 'private' || $scheme === 'both') {
      $schemes_to_copy[] = 'private';
    }

    $batch = [
      'title' => $this->t('Copying files to OCI Object Storage'),
      'operations' => [],
      'finished' => '\Drupal\oci_osfs\Form\OciOsfsActionsForm::batchFinished',
    ];

    foreach ($schemes_to_copy as $copy_scheme) {
      // Get all managed files for this scheme
      $query = \Drupal::entityQuery('file')
        ->condition('uri', $copy_scheme . '://%', 'LIKE')
        ->accessCheck(FALSE);

      $fids = $query->execute();

      foreach ($fids as $fid) {
        $batch['operations'][] = [
          '\Drupal\oci_osfs\Form\OciOsfsActionsForm::batchCopyFile',
          [$fid, $copy_scheme],
        ];
      }
    }

    if (empty($batch['operations'])) {
      $this->messenger()->addWarning($this->t('No files found to copy.'));
      return;
    }

    batch_set($batch);
  }

  /**
   * Batch operation to copy a single file.
   */
  public static function batchCopyFile($fid, $scheme, &$context) {
    $file = File::load($fid);

    if (!$file) {
      $context['results']['errors'][] = t('File @fid not found', ['@fid' => $fid]);
      return;
    }

    $source_uri = $file->getFileUri();

    // Build OCI URI
    $target = str_replace($scheme . '://', '', $source_uri);
    $oci_uri = 'oci://' . $target;

    try {
      // Check if source exists
      if (!file_exists($source_uri)) {
        $context['results']['errors'][] = t('Source file does not exist: @uri', ['@uri' => $source_uri]);
        return;
      }

      // Check if already exists in OCI
      if (file_exists($oci_uri)) {
        $context['results']['skipped'][] = $source_uri;
        $context['message'] = t('Skipped (already exists): @file', ['@file' => $source_uri]);
        return;
      }

      // Copy file
      $directory = dirname($oci_uri);
      \Drupal::service('file_system')->prepareDirectory($directory, FileSystemInterface::CREATE_DIRECTORY);

      if (copy($source_uri, $oci_uri)) {
        $context['results']['copied'][] = $source_uri;
        $context['message'] = t('Copied: @file', ['@file' => $source_uri]);
      }
      else {
        $context['results']['errors'][] = t('Failed to copy: @uri', ['@uri' => $source_uri]);
      }
    }
    catch (\Exception $e) {
      $context['results']['errors'][] = t('Error copying @uri: @error', [
        '@uri' => $source_uri,
        '@error' => $e->getMessage(),
      ]);
    }
  }

  /**
   * Batch finished callback.
   */
  public static function batchFinished($success, $results, $operations) {
    $messenger = \Drupal::messenger();

    if ($success) {
      $copied = isset($results['copied']) ? count($results['copied']) : 0;
      $skipped = isset($results['skipped']) ? count($results['skipped']) : 0;
      $errors = isset($results['errors']) ? count($results['errors']) : 0;

      $messenger->addStatus(t('File copy completed.'));
      $messenger->addStatus(t('Copied: @count files', ['@count' => $copied]));

      if ($skipped > 0) {
        $messenger->addWarning(t('Skipped: @count files (already exist in OCI)', ['@count' => $skipped]));
      }

      if ($errors > 0) {
        $messenger->addError(t('Errors: @count files failed to copy', ['@count' => $errors]));
        foreach ($results['errors'] as $error) {
          $messenger->addError($error);
        }
      }
    }
    else {
      $messenger->addError(t('File copy failed.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Default submit handler - individual actions have their own handlers
  }

}
