<?php

namespace Drupal\oci_osfs\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

class OciOsfsSettingsForm extends ConfigFormBase {

  public function getFormId(): string {
    return 'oci_osfs_settings_form';
  }

  protected function getEditableConfigNames(): array {
    return ['oci_osfs.settings'];
  }

  public function buildForm(array $form, FormStateInterface $form_state): array {
    $c = $this->config('oci_osfs.settings');

    // Credentials section
    $form['credentials'] = [
      '#type' => 'details',
      '#title' => $this->t('AWS Credentials (S3-compatible)'),
      '#open' => TRUE,
      '#description' => $this->t('Configure your OCI Customer Secret Keys for S3-compatible access to Object Storage.'),
    ];

    $form['credentials']['use_settings_file'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use credentials from settings.php instead of form'),
      '#default_value' => $c->get('use_settings_file') ?? TRUE,
      '#description' => $this->t('When checked, credentials are read from settings.php ($settings[\'oci_osfs.customer_keys\']). When unchecked, use the fields below.'),
    ];

    // Show which credentials are in settings.php
    $settings = \Drupal\Core\Site\Settings::get('oci_osfs.customer_keys', []);
    $has_settings_keys = !empty($settings['access_key']);

    if ($has_settings_keys) {
      $form['credentials']['settings_status'] = [
        '#type' => 'item',
        '#markup' => '<div class="messages messages--status">' .
          $this->t('Customer Secret Keys found in settings.php') .
          '</div>',
        '#states' => [
          'visible' => [
            ':input[name="use_settings_file"]' => ['checked' => TRUE],
          ],
        ],
      ];
    } else {
      $form['credentials']['settings_status'] = [
        '#type' => 'item',
        '#markup' => '<div class="messages messages--warning">' .
          $this->t('No Customer Secret Keys found in settings.php. Either configure them below or add to settings.php.') .
          '</div>',
        '#states' => [
          'visible' => [
            ':input[name="use_settings_file"]' => ['checked' => TRUE],
          ],
        ],
      ];
    }

    $form['credentials']['access_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Access Key ID'),
      '#default_value' => $c->get('access_key'),
      '#description' => $this->t('Your OCI Customer Secret Key Access Key ID'),
      '#states' => [
        'visible' => [
          ':input[name="use_settings_file"]' => ['checked' => FALSE],
        ],
        'required' => [
          ':input[name="use_settings_file"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $form['credentials']['secret_key'] = [
      '#type' => 'password',
      '#title' => $this->t('Secret Access Key'),
      '#default_value' => $c->get('secret_key'),
      '#description' => $this->t('Your OCI Customer Secret Key. Note: This will be stored in plain text in the database.'),
      '#attributes' => ['autocomplete' => 'off'],
      '#states' => [
        'visible' => [
          ':input[name="use_settings_file"]' => ['checked' => FALSE],
        ],
        'required' => [
          ':input[name="use_settings_file"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $form['credentials']['secret_key_note'] = [
      '#type' => 'item',
      '#markup' => '<div class="description">' .
        $this->t('<strong>Security Note:</strong> For production environments, it is recommended to store credentials in settings.php instead of the database. See the module README for instructions.') .
        '</div>',
      '#states' => [
        'visible' => [
          ':input[name="use_settings_file"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $form['region'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Region'),
      '#default_value' => $c->get('region'),
      '#required' => TRUE,
    ];

    $form['namespace'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Object Storage namespace'),
      '#default_value' => $c->get('namespace'),
      '#required' => TRUE,
    ];

    $form['bucket'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Bucket'),
      '#default_value' => $c->get('bucket'),
      '#required' => TRUE,
    ];

    $form['root_prefix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Root prefix'),
      '#default_value' => $c->get('root_prefix'),
      '#description' => $this->t('Optional base prefix inside the bucket.'),
    ];

    $form['public_prefix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Public prefix'),
      '#default_value' => $c->get('public_prefix') ?: 'public',
      '#required' => TRUE,
    ];

    $form['private_prefix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Private prefix'),
      '#default_value' => $c->get('private_prefix') ?: 'private',
      '#required' => TRUE,
    ];

    $form['allowed_prefixes'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Allowed prefixes (one per line)'),
      '#default_value' => implode("\n", (array) $c->get('allowed_prefixes')),
      '#description' => $this->t('If set, access is restricted to these prefixes relative to the bucket root.'),
    ];

    $form['override_public'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use OCI for public://'),
      '#default_value' => (bool) $c->get('override_public'),
    ];

    $form['override_private'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use OCI for private://'),
      '#default_value' => (bool) $c->get('override_private'),
    ];

    $form['public_delivery'] = [
      '#type' => 'select',
      '#title' => $this->t('Public delivery'),
      '#options' => [
        'direct' => $this->t('Direct URL'),
        'drupal' => $this->t('Serve through Drupal'),
      ],
      '#default_value' => $c->get('public_delivery') ?: 'direct',
    ];

    $form['private_delivery'] = [
      '#type' => 'select',
      '#title' => $this->t('Private delivery'),
      '#options' => [
        'drupal' => $this->t('Serve through Drupal'),
        'signed' => $this->t('Signed URL'),
      ],
      '#default_value' => $c->get('private_delivery') ?: 'drupal',
    ];

    $form['signed_ttl'] = [
      '#type' => 'number',
      '#title' => $this->t('Signed URL TTL (seconds)'),
      '#default_value' => (int) ($c->get('signed_ttl') ?: 300),
      '#min' => 60,
      '#max' => 86400,
      '#states' => [
        'visible' => [
          ':input[name="private_delivery"]' => ['value' => 'signed'],
        ],
      ],
    ];

    $form['metadata_cache_max_age'] = [
      '#type' => 'number',
      '#title' => $this->t('Metadata cache TTL (seconds)'),
      '#default_value' => (int) ($c->get('metadata_cache_max_age') ?: 300),
      '#min' => 0,
      '#max' => 3600,
    ];

    return parent::buildForm($form, $form_state);
  }

  public function validateForm(array &$form, FormStateInterface $form_state): void {
    foreach (['root_prefix', 'public_prefix', 'private_prefix'] as $k) {
      $v = (string) $form_state->getValue($k);
      if (str_contains($v, '..')) {
        $form_state->setErrorByName($k, $this->t('Prefix must not contain "..".'));
      }
    }
    parent::validateForm($form, $form_state);
  }

  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $allowed = array_filter(array_map('trim', explode("\n", (string) $form_state->getValue('allowed_prefixes'))));

    $config = $this->config('oci_osfs.settings');

    // Save credential settings
    $config->set('use_settings_file', (bool) $form_state->getValue('use_settings_file'));

    // Only save credentials if not using settings.php
    if (!$form_state->getValue('use_settings_file')) {
      $config->set('access_key', trim((string) $form_state->getValue('access_key')));

      // Only update secret key if a new value was provided (password field)
      $secret_key = $form_state->getValue('secret_key');
      if (!empty($secret_key)) {
        $config->set('secret_key', trim((string) $secret_key));
      }
    }

    $config
      ->set('region', trim((string) $form_state->getValue('region')))
      ->set('namespace', trim((string) $form_state->getValue('namespace')))
      ->set('bucket', trim((string) $form_state->getValue('bucket')))
      ->set('root_prefix', trim((string) $form_state->getValue('root_prefix'), '/'))
      ->set('public_prefix', trim((string) $form_state->getValue('public_prefix'), '/'))
      ->set('private_prefix', trim((string) $form_state->getValue('private_prefix'), '/'))
      ->set('allowed_prefixes', array_values($allowed))
      ->set('override_public', (bool) $form_state->getValue('override_public'))
      ->set('override_private', (bool) $form_state->getValue('override_private'))
      ->set('public_delivery', $form_state->getValue('public_delivery'))
      ->set('private_delivery', $form_state->getValue('private_delivery'))
      ->set('signed_ttl', (int) $form_state->getValue('signed_ttl'))
      ->set('metadata_cache_max_age', (int) $form_state->getValue('metadata_cache_max_age'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
