<?php

namespace Drupal\oci_osfs\Service;

use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;

/**
 * Logging service for OCI OSFS operations.
 */
class OciLogger {

  /**
   * The logger channel.
   */
  protected LoggerChannelInterface $logger;

  /**
   * Constructs an OciLogger object.
   *
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger factory.
   */
  public function __construct(LoggerChannelFactoryInterface $loggerFactory) {
    $this->logger = $loggerFactory->get('oci_osfs');
  }

  /**
   * Logs an operation.
   *
   * @param string $operation
   *   The operation name.
   * @param array $context
   *   Additional context data.
   */
  public function logOperation(string $operation, array $context = []): void {
    $this->logger->info('Operation: @operation', [
      '@operation' => $operation,
    ] + $context);
  }

  /**
   * Logs an error.
   *
   * @param string $message
   *   The error message.
   * @param \Throwable $exception
   *   The exception.
   * @param array $context
   *   Additional context data.
   */
  public function logError(string $message, \Throwable $exception, array $context = []): void {
    $this->logger->error('@message: @error in @file:@line', [
      '@message' => $message,
      '@error' => $exception->getMessage(),
      '@file' => $exception->getFile(),
      '@line' => $exception->getLine(),
    ] + $context);
  }

  /**
   * Logs a security event.
   *
   * @param string $event
   *   The security event description.
   * @param array $context
   *   Additional context data.
   */
  public function logSecurityEvent(string $event, array $context = []): void {
    $this->logger->warning('Security event: @event', [
      '@event' => $event,
      'ip' => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
      'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'unknown',
    ] + $context);
  }

  /**
   * Logs a performance metric.
   *
   * @param string $metric
   *   The metric name.
   * @param float $value
   *   The metric value.
   * @param array $context
   *   Additional context data.
   */
  public function logPerformance(string $metric, float $value, array $context = []): void {
    $this->logger->debug('Performance: @metric = @value', [
      '@metric' => $metric,
      '@value' => $value,
    ] + $context);
  }

  /**
   * Logs an access attempt.
   *
   * @param string $uri
   *   The URI being accessed.
   * @param bool $granted
   *   Whether access was granted.
   * @param array $context
   *   Additional context data.
   */
  public function logAccess(string $uri, bool $granted, array $context = []): void {
    $level = $granted ? 'debug' : 'warning';
    $status = $granted ? 'granted' : 'denied';

    $this->logger->log($level, 'Access @status: @uri', [
      '@status' => $status,
      '@uri' => $uri,
    ] + $context);
  }

  /**
   * Logs SDK interaction.
   *
   * @param string $method
   *   The SDK method called.
   * @param array $params
   *   The parameters (sensitive data will be redacted).
   * @param array $context
   *   Additional context data.
   */
  public function logSdkCall(string $method, array $params = [], array $context = []): void {
    // Redact sensitive parameters.
    $safe_params = $this->redactSensitiveData($params);

    $this->logger->debug('SDK call: @method with params: @params', [
      '@method' => $method,
      '@params' => json_encode($safe_params),
    ] + $context);
  }

  /**
   * Redacts sensitive data from parameters.
   *
   * @param array $data
   *   The data array.
   *
   * @return array
   *   The redacted data array.
   */
  protected function redactSensitiveData(array $data): array {
    $sensitive_keys = [
      'private_key',
      'private_key_path',
      'passphrase',
      'password',
      'secret',
      'token',
      'api_key',
    ];

    $redacted = [];
    foreach ($data as $key => $value) {
      if (in_array(strtolower($key), $sensitive_keys, TRUE)) {
        $redacted[$key] = '[REDACTED]';
      }
      elseif (is_array($value)) {
        $redacted[$key] = $this->redactSensitiveData($value);
      }
      else {
        $redacted[$key] = $value;
      }
    }

    return $redacted;
  }

}
