<?php

namespace Drupal\oci_osfs\Service;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;

class OciMetadataCache {

  public function __construct(
    protected CacheBackendInterface $cache,
    protected ConfigFactoryInterface $configFactory
  ) {}

  protected function ttl(): int {
    $c = $this->configFactory->get('oci_osfs.settings');
    return max(0, (int) ($c->get('metadata_cache_max_age') ?? 300));
  }

  protected function cid(string $bucket, string $key): string {
    return "oci_osfs:meta:" . hash('sha256', $bucket . '|' . $key);
  }

  public function get(string $bucket, string $key): ?array {
    $item = $this->cache->get($this->cid($bucket, $key));
    return $item ? $item->data : NULL;
  }

  public function set(string $bucket, string $key, array $data): void {
    $ttl = $this->ttl();
    if ($ttl === 0) {
      return;
    }
    $this->cache->set($this->cid($bucket, $key), $data, time() + $ttl);
  }

  public function delete(string $bucket, string $key): void {
    $this->cache->delete($this->cid($bucket, $key));
  }

  /**
   * Clear all metadata cache entries.
   */
  public function deleteAll(): void {
    $this->cache->deleteMultiple([]);
    // Alternative: invalidate by tag if using cache tags
    // But since we're using hash-based cache IDs, we'll clear all
    $this->cache->invalidateAll();
  }

}
