<?php

namespace Drupal\oci_osfs\Service;

use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Registers and provides access to the OCI SDK's own PHP stream wrapper.
 *
 * The official SDK includes a Stream Wrapper over Objects feature.
 * We register it under a private scheme to avoid conflicts with Drupal's scheme.
 */
class OciPhpStreamBridge {

  protected bool $registered = FALSE;

  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    protected OciAuthFactory $authFactory
  ) {}

  public function ensureRegistered(): void {
    if ($this->registered) {
      return;
    }

    $c = $this->configFactory->get('oci_osfs.settings');
    $auth = $this->authFactory->createAuth();

    $region = (string) $c->get('region');
    $namespace = (string) $c->get('namespace');
    $bucket = (string) $c->get('bucket');

    if ($region === '' || $namespace === '' || $bucket === '') {
      throw new \RuntimeException('OCI stream bridge not configured.');
    }

    // PSEUDOCODE:
    //
    // use Oracle\Oci\ObjectStorage\StreamWrapper\StreamWrapper;
    //
    // StreamWrapper::register('oci_native', [
    //   'region' => $region,
    //   'namespace' => $namespace,
    //   'bucket' => $bucket,
    //   'auth' => $auth,
    // ]);
    //
    // The official SDK release notes mention Stream Wrapper support.

    $this->registered = TRUE;
  }

  public function buildNativeUri(string $objectKey): string {
    $this->ensureRegistered();
    return 'oci_native://' . ltrim($objectKey, '/');
  }

}
