<?php

namespace Drupal\oci_osfs\Service;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Rate limiting service for OCI operations.
 *
 * Prevents abuse and throttles requests to protect API quotas.
 */
class OciRateLimiter {

  /**
   * Default rate limit: operations per minute.
   */
  const DEFAULT_RATE_LIMIT = 100;

  /**
   * Window size in seconds.
   */
  const WINDOW_SIZE = 60;

  public function __construct(
    protected CacheBackendInterface $cache,
    protected ConfigFactoryInterface $configFactory,
    protected OciLogger $logger,
  ) {}

  /**
   * Checks if an operation is allowed under rate limits.
   *
   * @param string $operation
   *   The operation type.
   * @param string $identifier
   *   The identifier (IP, user ID, etc.).
   *
   * @return bool
   *   TRUE if allowed, FALSE if rate limited.
   */
  public function isAllowed(string $operation, string $identifier): bool {
    $limit = $this->getLimit($operation);
    if ($limit <= 0) {
      // Rate limiting disabled.
      return TRUE;
    }

    $key = $this->getCacheKey($operation, $identifier);
    $cache_item = $this->cache->get($key);

    $current_count = $cache_item ? (int) $cache_item->data : 0;

    if ($current_count >= $limit) {
      $this->logger->logSecurityEvent('Rate limit exceeded', [
        'operation' => $operation,
        'identifier' => $identifier,
        'count' => $current_count,
        'limit' => $limit,
      ]);
      return FALSE;
    }

    // Increment counter.
    $this->cache->set($key, $current_count + 1, time() + self::WINDOW_SIZE);

    return TRUE;
  }

  /**
   * Records an operation for rate limiting.
   *
   * @param string $operation
   *   The operation type.
   * @param string $identifier
   *   The identifier.
   */
  public function recordOperation(string $operation, string $identifier): void {
    $this->isAllowed($operation, $identifier);
  }

  /**
   * Gets the rate limit for an operation.
   *
   * @param string $operation
   *   The operation type.
   *
   * @return int
   *   The rate limit (operations per minute).
   */
  protected function getLimit(string $operation): int {
    $config = $this->configFactory->get('oci_osfs.settings');
    $limits = $config->get('rate_limits') ?: [];

    return $limits[$operation] ?? self::DEFAULT_RATE_LIMIT;
  }

  /**
   * Generates cache key for rate limiting.
   *
   * @param string $operation
   *   The operation type.
   * @param string $identifier
   *   The identifier.
   *
   * @return string
   *   The cache key.
   */
  protected function getCacheKey(string $operation, string $identifier): string {
    return 'oci_osfs:rate_limit:' . hash('sha256', $operation . ':' . $identifier);
  }

  /**
   * Resets rate limit for an identifier.
   *
   * @param string $operation
   *   The operation type.
   * @param string $identifier
   *   The identifier.
   */
  public function reset(string $operation, string $identifier): void {
    $key = $this->getCacheKey($operation, $identifier);
    $this->cache->delete($key);
  }

  /**
   * Gets current count for an identifier.
   *
   * @param string $operation
   *   The operation type.
   * @param string $identifier
   *   The identifier.
   *
   * @return int
   *   Current operation count.
   */
  public function getCurrentCount(string $operation, string $identifier): int {
    $key = $this->getCacheKey($operation, $identifier);
    $cache_item = $this->cache->get($key);

    return $cache_item ? (int) $cache_item->data : 0;
  }

}
