<?php

namespace Drupal\oci_osfs\Service;

use Aws\S3\S3Client;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\oci_osfs\Service\OciAuthFactory;

/**
 * Factory for creating S3 clients for OCI Object Storage.
 */
class S3ClientFactory {

  protected ConfigFactoryInterface $configFactory;
  protected OciAuthFactory $authFactory;
  protected ?S3Client $client = NULL;

  public function __construct(
    ConfigFactoryInterface $configFactory,
    OciAuthFactory $authFactory
  ) {
    $this->configFactory = $configFactory;
    $this->authFactory = $authFactory;
  }

  /**
   * Get or create S3 client.
   */
  public function getClient(): S3Client {
    if ($this->client === NULL) {
      $this->client = $this->createClient();
    }
    return $this->client;
  }

  /**
   * Create S3 client configured for OCI Object Storage.
   */
  protected function createClient(): S3Client {
    $config = $this->configFactory->get('oci_osfs.settings');
    $auth = $this->authFactory->createAuth();

    if ($auth['type'] !== 'customer_secret_keys') {
      throw new \RuntimeException('S3 client only supports Customer Secret Keys authentication');
    }

    $region = $config->get('region');
    $namespace = $config->get('namespace');

    // OCI S3-compatible endpoint
    $endpoint = "https://{$namespace}.compat.objectstorage.{$region}.oraclecloud.com";

    return new S3Client([
      'version' => 'latest',
      'region' => $region,
      'endpoint' => $endpoint,
      'use_path_style_endpoint' => TRUE,
      'credentials' => [
        'key' => $auth['access_key'],
        'secret' => $auth['secret_key'],
      ],
    ]);
  }

}
