<?php

namespace Drupal\Tests\oci_osfs\Unit;

use Drupal\oci_osfs\Service\OciSecurityValidator;
use Drupal\Tests\UnitTestCase;

/**
 * Tests for the OCI Security Validator service.
 *
 * @group oci_osfs
 * @coversDefaultClass \Drupal\oci_osfs\Service\OciSecurityValidator
 */
class OciSecurityValidatorTest extends UnitTestCase {

  /**
   * The security validator.
   *
   * @var \Drupal\oci_osfs\Service\OciSecurityValidator
   */
  protected OciSecurityValidator $validator;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->validator = new OciSecurityValidator();
  }

  /**
   * Tests path validation.
   *
   * @covers ::validatePath
   */
  public function testValidatePath(): void {
    // Valid paths should not throw.
    $this->validator->validatePath('valid/path/to/file.txt');
    $this->validator->validatePath('another-valid_path.jpg');

    // Path traversal should throw.
    $this->expectException(\InvalidArgumentException::class);
    $this->validator->validatePath('../../../etc/passwd');
  }

  /**
   * Tests null byte detection.
   *
   * @covers ::validatePath
   */
  public function testNullByteDetection(): void {
    $this->expectException(\InvalidArgumentException::class);
    $this->expectExceptionMessage('Null byte detected');
    $this->validator->validatePath("file\0.txt");
  }

  /**
   * Tests extension validation.
   *
   * @covers ::validateExtension
   */
  public function testValidateExtension(): void {
    // Safe extensions should not throw.
    $this->validator->validateExtension('image.jpg');
    $this->validator->validateExtension('document.pdf');

    // Dangerous extensions should throw.
    $this->expectException(\InvalidArgumentException::class);
    $this->validator->validateExtension('malware.php');
  }

  /**
   * Tests bucket name validation.
   *
   * @covers ::validateBucketName
   */
  public function testValidateBucketName(): void {
    // Valid bucket names.
    $this->validator->validateBucketName('my-bucket');
    $this->validator->validateBucketName('bucket123');

    // Invalid: starts with hyphen.
    $this->expectException(\InvalidArgumentException::class);
    $this->validator->validateBucketName('-invalid');
  }

  /**
   * Tests namespace validation.
   *
   * @covers ::validateNamespace
   */
  public function testValidateNamespace(): void {
    // Valid namespace.
    $this->validator->validateNamespace('validnamespace123');

    // Invalid: contains hyphen.
    $this->expectException(\InvalidArgumentException::class);
    $this->validator->validateNamespace('invalid-namespace');
  }

  /**
   * Tests region validation.
   *
   * @covers ::validateRegion
   */
  public function testValidateRegion(): void {
    // Valid regions.
    $this->validator->validateRegion('us-phoenix-1');
    $this->validator->validateRegion('eu-frankfurt-1');
    $this->validator->validateRegion('ap-sydney-1');

    // Invalid region format.
    $this->expectException(\InvalidArgumentException::class);
    $this->validator->validateRegion('invalid-region');
  }

  /**
   * Tests OCID validation.
   *
   * @covers ::validateOcid
   */
  public function testValidateOcid(): void {
    // Valid OCID.
    $this->validator->validateOcid('ocid1.tenancy.oc1..aaaaaaaabbbbbbbcccccccddddddd', 'tenancy');
    $this->validator->validateOcid('ocid1.user.oc1..aaaaaaaabbbbbbbcccccccddddddd', 'user');

    // Invalid OCID.
    $this->expectException(\InvalidArgumentException::class);
    $this->validator->validateOcid('invalid-ocid', 'tenancy');
  }

  /**
   * Tests fingerprint validation.
   *
   * @covers ::validateFingerprint
   */
  public function testValidateFingerprint(): void {
    // Valid fingerprint.
    $this->validator->validateFingerprint('aa:bb:cc:dd:ee:ff:00:11:22:33:44:55:66:77:88:99');

    // Invalid fingerprint.
    $this->expectException(\InvalidArgumentException::class);
    $this->validator->validateFingerprint('invalid-fingerprint');
  }

  /**
   * Tests TTL validation.
   *
   * @covers ::validateTtl
   */
  public function testValidateTtl(): void {
    // Valid TTL.
    $this->validator->validateTtl(300);
    $this->validator->validateTtl(3600);

    // Invalid: too low.
    $this->expectException(\InvalidArgumentException::class);
    $this->validator->validateTtl(30);
  }

  /**
   * Tests prefix sanitization.
   *
   * @covers ::sanitizePrefix
   */
  public function testSanitizePrefix(): void {
    $this->assertEquals('public', $this->validator->sanitizePrefix('  /public/  '));
    $this->assertEquals('my/prefix', $this->validator->sanitizePrefix('../my/prefix'));
    $this->assertEquals('safe', $this->validator->sanitizePrefix("safe\0byte"));
  }

}
