<?php

namespace Drupal\oci_osfs\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Site\Settings;
use Drupal\oci_osfs\Exception\OciAuthenticationException;

/**
 * Builds auth configuration for the official OCI SDK.
 *
 * Secrets are read from settings.php / env.
 */
class OciAuthFactory {

  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    protected Settings $settings,
    protected OciSecurityValidator $securityValidator,
    protected OciLogger $logger,
  ) {}

  /**
   * Returns an array describing auth to be used by the SDK client factory.
   *
   * Auto-detects authentication method based on available credentials:
   * 1. Customer Secret Keys from config OR settings.php
   * 2. API Key (if OCI_TENANCY_OCID, OCI_USER_OCID, etc. are set in settings.php)
   * 3. Instance Principals (if running on OCI Compute)
   */
  public function createAuth(): array {
    // Priority 1: Check if Customer Secret Keys are available
    $config = $this->configFactory->get('oci_osfs.settings');
    $use_settings_file = $config->get('use_settings_file') ?? TRUE;

    $customer_keys = [];

    if ($use_settings_file) {
      // Read from settings.php
      $customer_keys = $this->settings->get('oci_osfs.customer_keys', []);
    } else {
      // Read from config (admin UI)
      $access_key = $config->get('access_key');
      $secret_key = $config->get('secret_key');
      if (!empty($access_key) && !empty($secret_key)) {
        $customer_keys = [
          'access_key' => $access_key,
          'secret_key' => $secret_key,
        ];
      }
    }

    if (!empty($customer_keys['access_key']) && !empty($customer_keys['secret_key'])) {
      return $this->createCustomerSecretKeyAuth($customer_keys);
    }

    // Priority 2: Check if API Key credentials are available
    $api_key = $this->settings->get('oci_osfs.api_key', []);
    if (!empty($api_key['tenancy_ocid'])) {
      return $this->createApiKeyAuth();
    }

    // Priority 3: Fall back to Instance Principals (if running on OCI)
    // This will fail if not on OCI Compute instance
    $this->logger->logOperation('auth_instance_principals');
    return ['type' => 'instance_principals'];
  }

  /**
   * Creates API key authentication configuration.
   *
   * @return array
   *   Authentication configuration.
   *
   * @throws \Drupal\oci_osfs\Exception\OciAuthenticationException
   */
  protected function createApiKeyAuth(): array {
    $api = $this->settings->get('oci_osfs.api_key', []);

    // Validate required fields (private_key OR private_key_path is required).
    foreach (['tenancy_ocid', 'user_ocid', 'fingerprint'] as $req) {
      if (empty($api[$req])) {
        $this->logger->logSecurityEvent('Missing API key setting: ' . $req);
        throw new OciAuthenticationException("Missing OCI API Key setting: $req");
      }
    }

    // Require either private_key or private_key_path.
    if (empty($api['private_key']) && empty($api['private_key_path'])) {
      $this->logger->logSecurityEvent('Missing private key: either private_key or private_key_path required');
      throw new OciAuthenticationException("Missing OCI API Key setting: private_key or private_key_path required");
    }

    try {
      // Validate OCIDs.
      $this->securityValidator->validateOcid($api['tenancy_ocid'], 'tenancy');
      $this->securityValidator->validateOcid($api['user_ocid'], 'user');
      $this->securityValidator->validateFingerprint($api['fingerprint']);

      $auth = [
        'type' => 'api_key',
        'tenancy_ocid' => $api['tenancy_ocid'],
        'user_ocid' => $api['user_ocid'],
        'fingerprint' => $api['fingerprint'],
        'passphrase' => $api['passphrase'] ?? NULL,
      ];

      // Option 1: Use private key content directly (preferred).
      if (!empty($api['private_key'])) {
        // Validate private key format.
        if (strpos($api['private_key'], '-----BEGIN') === FALSE) {
          throw new OciAuthenticationException("Invalid private key format");
        }

        $auth['private_key'] = $api['private_key'];
        $this->logger->logOperation('auth_api_key_content');
      }
      // Option 2: Use private key file path.
      else {
        $key_path = $api['private_key_path'];
        if (!file_exists($key_path) || !is_readable($key_path)) {
          throw new OciAuthenticationException("Private key file not found or not readable: $key_path");
        }

        // Check file permissions (should not be world-readable).
        $perms = fileperms($key_path);
        if ($perms & 0004) {
          $this->logger->logSecurityEvent('Private key file is world-readable', [
            'path' => $key_path,
          ]);
        }

        $auth['private_key_path'] = $key_path;
        $this->logger->logOperation('auth_api_key_file');
      }

      return $auth;
    }
    catch (\InvalidArgumentException $e) {
      $this->logger->logError('Authentication validation failed', $e);
      throw new OciAuthenticationException('Invalid authentication configuration: ' . $e->getMessage(), 0, $e);
    }
  }

  /**
   * Creates Customer Secret Key authentication (S3-compatible).
   *
   * @param array $keys
   *   Customer secret keys configuration.
   *
   * @return array
   *   Authentication configuration.
   *
   * @throws \Drupal\oci_osfs\Exception\OciAuthenticationException
   */
  protected function createCustomerSecretKeyAuth(array $keys): array {
    // Validate required fields.
    if (empty($keys['access_key']) || empty($keys['secret_key'])) {
      throw new OciAuthenticationException("Customer Secret Keys require both access_key and secret_key");
    }

    $this->logger->logOperation('auth_customer_secret_keys');

    return [
      'type' => 'customer_secret_keys',
      'access_key' => $keys['access_key'],
      'secret_key' => $keys['secret_key'],
    ];
  }

}
