<?php

namespace Drupal\oci_osfs\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\oci_osfs\Service\S3ClientFactory;

/**
 * Generates public and signed URLs for OCI Object Storage.
 */
class OciUrlGenerator {

  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    protected S3ClientFactory $s3ClientFactory,
    protected OciPathResolver $pathResolver,
    protected $logger = NULL
  ) {
    if ($logger === NULL && \Drupal::hasService('oci_osfs.logger')) {
      $this->logger = \Drupal::service('oci_osfs.logger');
    }
  }

  /**
   * Build a direct public URL for an object.
   *
   * For S3-compatible authentication (Customer Secret Keys), this generates
   * a presigned URL that's valid for 7 days. For other auth methods, it
   * returns the native OCI Object Storage URL.
   *
   * Note: Presigned URLs are generated even if the file doesn't exist yet.
   * This is intentional to support Drupal's image style generation workflow
   * where derivatives are created on-demand.
   */
  public function publicUrl(string $scheme, string $target): string {
    $c = $this->configFactory->get('oci_osfs.settings');
    $bucket = (string) $c->get('bucket');
    $key = $this->pathResolver->toObjectKey($scheme, $target);

    // Use S3-compatible API with presigned URLs
    try {
      $client = $this->s3ClientFactory->getClient();

      // Generate presigned URL even if file doesn't exist yet
      // This supports Drupal's image style workflow where derivatives
      // are created on-demand when first accessed
      $cmd = $client->getCommand('GetObject', [
        'Bucket' => $bucket,
        'Key' => $key,
      ]);

      // Create presigned URL valid for 7 days (604800 seconds)
      $request = $client->createPresignedRequest($cmd, '+7 days');
      return (string) $request->getUri();
    }
    catch (\Exception $e) {
      if ($this->logger) {
        $this->logger->logError('publicUrl', $e->getMessage(), ['key' => $key]);
      }
      // Fall through to native URL as fallback
    }

    // Fallback: native OCI Object Storage URL (requires public bucket or PAR)
    $region = (string) $c->get('region');
    $namespace = (string) $c->get('namespace');
    return "https://objectstorage.$region.oraclecloud.com/n/$namespace/b/$bucket/o/" . rawurlencode($key);
  }

  /**
   * Build a signed URL using S3 presigned requests.
   */
  public function signedUrl(string $scheme, string $target, ?int $ttl = NULL): string {
    $c = $this->configFactory->get('oci_osfs.settings');
    $ttl = $ttl ?? (int) ($c->get('signed_ttl') ?: 300);
    $ttl = max(60, min(86400, $ttl));

    $bucket = (string) $c->get('bucket');
    $key = $this->pathResolver->toObjectKey($scheme, $target);

    try {
      $client = $this->s3ClientFactory->getClient();
      $cmd = $client->getCommand('GetObject', [
        'Bucket' => $bucket,
        'Key' => $key,
      ]);

      // Create presigned URL with custom TTL
      $request = $client->createPresignedRequest($cmd, "+{$ttl} seconds");
      return (string) $request->getUri();
    }
    catch (\Exception $e) {
      if ($this->logger) {
        $this->logger->logError('signedUrl', $e->getMessage(), ['key' => $key]);
      }
      throw new \RuntimeException('Failed to generate signed URL: ' . $e->getMessage());
    }
  }

}
