# Security Guidelines for OCI Object Storage File System

## Overview
This document outlines security best practices and features implemented in the OCI OSFS module.

## Security Features

### 1. Input Validation
- **Path Traversal Protection**: All paths are validated to prevent `../` attacks
- **Null Byte Detection**: Prevents null byte injection attacks
- **Control Character Filtering**: Removes control characters from paths
- **Extension Blocking**: Blocks dangerous file extensions (.php, .exe, etc.)

### 2. Authentication Security
- **OCID Validation**: Validates Oracle Cloud Identifiers format
- **Fingerprint Validation**: Ensures API key fingerprints match expected format
- **Private Key Protection**: Checks file permissions and warns if world-readable
- **Instance Principals**: Supports more secure instance-based authentication

### 3. Access Control
- **Prefix Allowlist**: Optional restriction to specific object prefixes
- **Scheme-based Access**: Separate handling for public:// and private:// schemes
- **Permission Checks**: Drupal permission integration for private files

### 4. Rate Limiting
- **Per-Operation Limits**: Configurable rate limits for different operations
- **IP-based Tracking**: Tracks operations by client identifier
- **Automatic Throttling**: Prevents API quota exhaustion

### 5. Logging and Monitoring
- **Security Event Logging**: Logs suspicious activities and access denials
- **Operation Logging**: Tracks all OCI operations for audit trails
- **Sensitive Data Redaction**: Automatically redacts credentials in logs
- **Error Tracking**: Comprehensive error logging with context

## Configuration Best Practices

### 1. API Key Storage
```php
// settings.php - NEVER commit to version control
$settings['oci_osfs.api_key'] = [
  'tenancy_ocid' => getenv('OCI_TENANCY_OCID'),
  'user_ocid' => getenv('OCI_USER_OCID'),
  'fingerprint' => getenv('OCI_FINGERPRINT'),
  'private_key_path' => getenv('OCI_PRIVATE_KEY_PATH'),
  'passphrase' => getenv('OCI_PRIVATE_KEY_PASSPHRASE'),
];
```

**Important:**
- Store credentials in environment variables
- Never commit `.env` files to git
- Add `.env` to `.gitignore`
- Use different credentials per environment

### 2. Private Key File Permissions
```bash
# Set proper permissions on private key file
chmod 600 /path/to/private-key.pem
chown www-data:www-data /path/to/private-key.pem
```

**Important:**
- Private key should NOT be world-readable
- Should be owned by web server user
- Store outside web root if possible

### 3. Prefix Allowlist Configuration
Navigate to `/admin/config/media/oci-osfs` and configure allowed prefixes:

```
public
private
uploads
media
```

This restricts access to only these prefixes within your bucket.

### 4. Rate Limiting Configuration
Add to `config/install/oci_osfs.settings.yml`:

```yaml
rate_limits:
  read: 100          # 100 reads per minute per IP
  write: 50          # 50 writes per minute per IP
  delete: 20         # 20 deletes per minute per IP
  list: 30           # 30 directory listings per minute per IP
```

## Security Checklist

### Installation
- [ ] OCI PHP SDK installed via Composer
- [ ] Private key file has correct permissions (600)
- [ ] Credentials stored in environment variables
- [ ] `.env` file added to `.gitignore`
- [ ] Different credentials for dev/staging/prod

### Configuration
- [ ] Region, namespace, and bucket configured
- [ ] Authentication mode selected (prefer Instance Principals in production)
- [ ] Prefix allowlist configured (if needed)
- [ ] Public/private delivery methods configured appropriately
- [ ] Signed URL TTL set to minimum required value

### Operations
- [ ] Regular security audit logs reviewed
- [ ] Rate limits monitored and adjusted
- [ ] Cache TTL configured appropriately
- [ ] Backup authentication credentials stored securely
- [ ] Access permissions reviewed regularly

## Common Security Issues

### Issue: Private Key World-Readable
**Problem:** Private key file has permissions 644 or 666
**Solution:** `chmod 600 /path/to/private-key.pem`
**Impact:** HIGH - Anyone on server can read your credentials

### Issue: Credentials in settings.php
**Problem:** Hard-coded credentials in settings.php
**Solution:** Use environment variables with `getenv()`
**Impact:** HIGH - Credentials may be committed to version control

### Issue: No Prefix Allowlist
**Problem:** No prefix restrictions configured
**Solution:** Configure allowed prefixes in module settings
**Impact:** MEDIUM - Users could access unintended bucket areas

### Issue: Long Signed URL TTL
**Problem:** Signed URLs valid for 24 hours
**Solution:** Reduce TTL to minimum required (e.g., 5-15 minutes)
**Impact:** MEDIUM - Leaked URLs remain valid longer

### Issue: Rate Limiting Disabled
**Problem:** Rate limits set to 0 or very high values
**Solution:** Configure appropriate limits (e.g., 100/minute)
**Impact:** MEDIUM - API quota exhaustion, DoS potential

## Incident Response

### If Credentials Are Compromised:

1. **Immediately rotate credentials:**
   ```bash
   # Generate new API key in OCI Console
   # Update environment variables
   # Restart application
   ```

2. **Revoke old credentials:**
   - Delete compromised API key in OCI Console
   - Check OCI Audit logs for suspicious activity

3. **Review access logs:**
   ```bash
   # Check Drupal logs
   drush watchdog:show --type=oci_osfs

   # Check for security events
   drush watchdog:show --severity=Warning
   ```

4. **Update affected systems:**
   - Update all environments using compromised credentials
   - Notify security team
   - Document incident

### If Unauthorized Access Detected:

1. **Block source IP** (if applicable)
2. **Review security logs** for pattern analysis
3. **Check bucket activity** in OCI Console
4. **Tighten prefix allowlist** if needed
5. **Lower rate limits** temporarily
6. **Enable additional monitoring**

## Compliance Considerations

### GDPR / Data Privacy
- Configure appropriate data retention policies
- Enable logging for audit requirements
- Use private:// scheme for personal data
- Implement data deletion procedures

### PCI DSS
- Never store payment card data in OCI Object Storage via this module
- If needed, ensure proper encryption and access controls
- Implement additional logging and monitoring

### HIPAA
- Use encryption at rest and in transit
- Implement strict access controls
- Enable comprehensive audit logging
- Ensure BAA with Oracle Cloud

## Reporting Security Issues

If you discover a security vulnerability:

1. **DO NOT** open a public GitHub issue
2. Email security details to your organization's security team
3. Include:
   - Description of the vulnerability
   - Steps to reproduce
   - Potential impact
   - Suggested fix (if available)

## References

- [OCI Security Best Practices](https://docs.oracle.com/en-us/iaas/Content/Security/Concepts/security_guide.htm)
- [Drupal Security Team](https://www.drupal.org/drupal-security-team)
- [OWASP Top 10](https://owasp.org/www-project-top-ten/)
