<?php

namespace Drupal\oci_osfs\Service;

/**
 * Security validation service for OCI OSFS operations.
 *
 * Provides comprehensive input validation and security checks.
 */
class OciSecurityValidator {

  /**
   * Maximum allowed file path length.
   */
  const MAX_PATH_LENGTH = 1024;

  /**
   * Blocked file extensions for security.
   */
  const BLOCKED_EXTENSIONS = [
    'php', 'phtml', 'php3', 'php4', 'php5', 'php7', 'phps',
    'phar', 'exe', 'com', 'bat', 'cmd', 'sh', 'bash',
    'cgi', 'pl', 'py', 'rb', 'js', 'jar', 'war',
  ];

  /**
   * Validates a file path for security issues.
   *
   * @param string $path
   *   The path to validate.
   *
   * @throws \InvalidArgumentException
   *   If the path is invalid or contains security issues.
   */
  public function validatePath(string $path): void {
    // Check for empty path.
    if (trim($path) === '') {
      throw new \InvalidArgumentException('Path cannot be empty.');
    }

    // Check path length.
    if (strlen($path) > self::MAX_PATH_LENGTH) {
      throw new \InvalidArgumentException('Path exceeds maximum allowed length.');
    }

    // Check for path traversal attempts.
    if (str_contains($path, '..')) {
      throw new \InvalidArgumentException('Path traversal detected.');
    }

    // Check for null bytes.
    if (str_contains($path, "\0")) {
      throw new \InvalidArgumentException('Null byte detected in path.');
    }

    // Check for control characters.
    if (preg_match('/[\x00-\x1F\x7F]/', $path)) {
      throw new \InvalidArgumentException('Control characters detected in path.');
    }

    // Check for absolute paths that might escape.
    if (str_starts_with($path, '/') && str_contains($path, '//')) {
      throw new \InvalidArgumentException('Invalid absolute path format.');
    }
  }

  /**
   * Validates a file extension.
   *
   * @param string $filename
   *   The filename to validate.
   * @param array $additional_blocked
   *   Additional extensions to block.
   *
   * @throws \InvalidArgumentException
   *   If the extension is blocked.
   */
  public function validateExtension(string $filename, array $additional_blocked = []): void {
    $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
    $blocked = array_merge(self::BLOCKED_EXTENSIONS, $additional_blocked);

    if (in_array($ext, $blocked, TRUE)) {
      throw new \InvalidArgumentException("File extension '.$ext' is not allowed for security reasons.");
    }
  }

  /**
   * Sanitizes a prefix string.
   *
   * @param string $prefix
   *   The prefix to sanitize.
   *
   * @return string
   *   The sanitized prefix.
   */
  public function sanitizePrefix(string $prefix): string {
    // Remove leading/trailing slashes and whitespace.
    $prefix = trim($prefix, "/ \t\n\r\0\x0B");

    // Remove any path traversal attempts.
    $prefix = str_replace('..', '', $prefix);

    // Remove control characters.
    $prefix = preg_replace('/[\x00-\x1F\x7F]/', '', $prefix);

    return $prefix;
  }

  /**
   * Validates bucket name according to OCI specifications.
   *
   * @param string $bucket
   *   The bucket name to validate.
   *
   * @throws \InvalidArgumentException
   *   If the bucket name is invalid.
   */
  public function validateBucketName(string $bucket): void {
    if (trim($bucket) === '') {
      throw new \InvalidArgumentException('Bucket name cannot be empty.');
    }

    // OCI bucket names: 1-256 chars, alphanumeric + hyphens.
    if (!preg_match('/^[a-zA-Z0-9][a-zA-Z0-9\-]{0,255}$/', $bucket)) {
      throw new \InvalidArgumentException('Invalid bucket name format.');
    }

    // Cannot start or end with hyphen.
    if (str_starts_with($bucket, '-') || str_ends_with($bucket, '-')) {
      throw new \InvalidArgumentException('Bucket name cannot start or end with hyphen.');
    }
  }

  /**
   * Validates namespace according to OCI specifications.
   *
   * @param string $namespace
   *   The namespace to validate.
   *
   * @throws \InvalidArgumentException
   *   If the namespace is invalid.
   */
  public function validateNamespace(string $namespace): void {
    if (trim($namespace) === '') {
      throw new \InvalidArgumentException('Namespace cannot be empty.');
    }

    // OCI namespaces are alphanumeric.
    if (!preg_match('/^[a-zA-Z0-9]+$/', $namespace)) {
      throw new \InvalidArgumentException('Invalid namespace format.');
    }
  }

  /**
   * Validates region according to OCI specifications.
   *
   * @param string $region
   *   The region to validate.
   *
   * @throws \InvalidArgumentException
   *   If the region is invalid.
   */
  public function validateRegion(string $region): void {
    if (trim($region) === '') {
      throw new \InvalidArgumentException('Region cannot be empty.');
    }

    // OCI regions follow pattern like us-phoenix-1, eu-frankfurt-1.
    if (!preg_match('/^[a-z]{2}-[a-z]+-\d+$/', $region)) {
      throw new \InvalidArgumentException('Invalid region format.');
    }
  }

  /**
   * Validates OCID format.
   *
   * @param string $ocid
   *   The OCID to validate.
   * @param string $type
   *   The expected resource type (e.g., 'tenancy', 'user').
   *
   * @throws \InvalidArgumentException
   *   If the OCID is invalid.
   */
  public function validateOcid(string $ocid, string $type): void {
    if (trim($ocid) === '') {
      throw new \InvalidArgumentException('OCID cannot be empty.');
    }

    // OCID format: ocid1.<resource-type>.<realm>.[region].<unique-id>
    $pattern = '/^ocid1\.' . preg_quote($type, '/') . '\.[a-z0-9\-\.]+$/';
    if (!preg_match($pattern, $ocid)) {
      throw new \InvalidArgumentException("Invalid OCID format for type '{$type}'.");
    }
  }

  /**
   * Validates a file key fingerprint.
   *
   * @param string $fingerprint
   *   The fingerprint to validate.
   *
   * @throws \InvalidArgumentException
   *   If the fingerprint is invalid.
   */
  public function validateFingerprint(string $fingerprint): void {
    if (trim($fingerprint) === '') {
      throw new \InvalidArgumentException('Fingerprint cannot be empty.');
    }

    // Format: xx:xx:xx:xx:xx:xx:xx:xx:xx:xx:xx:xx:xx:xx:xx:xx
    if (!preg_match('/^([0-9a-f]{2}:){15}[0-9a-f]{2}$/i', $fingerprint)) {
      throw new \InvalidArgumentException('Invalid fingerprint format.');
    }
  }

  /**
   * Validates TTL value.
   *
   * @param int $ttl
   *   The TTL value in seconds.
   * @param int $min
   *   Minimum allowed value.
   * @param int $max
   *   Maximum allowed value.
   *
   * @throws \InvalidArgumentException
   *   If the TTL is out of bounds.
   */
  public function validateTtl(int $ttl, int $min = 60, int $max = 86400): void {
    if ($ttl < $min || $ttl > $max) {
      throw new \InvalidArgumentException("TTL must be between {$min} and {$max} seconds.");
    }
  }

}
