<?php

namespace Drupal\Tests\oci_osfs\Unit;

use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\oci_osfs\Service\OciLogger;
use Drupal\Tests\UnitTestCase;

/**
 * Tests for the OCI Logger service.
 *
 * @group oci_osfs
 * @coversDefaultClass \Drupal\oci_osfs\Service\OciLogger
 */
class OciLoggerTest extends UnitTestCase {

  /**
   * The logger service.
   *
   * @var \Drupal\oci_osfs\Service\OciLogger
   */
  protected OciLogger $ociLogger;

  /**
   * The mock logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $mockChannel;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->mockChannel = $this->createMock(LoggerChannelInterface::class);

    $factory = $this->createMock(LoggerChannelFactoryInterface::class);
    $factory->method('get')
      ->with('oci_osfs')
      ->willReturn($this->mockChannel);

    $this->ociLogger = new OciLogger($factory);
  }

  /**
   * Tests logging operations.
   *
   * @covers ::logOperation
   */
  public function testLogOperation(): void {
    $this->mockChannel->expects($this->once())
      ->method('info')
      ->with(
        'Operation: @operation',
        $this->callback(function ($context) {
          return isset($context['@operation']) && $context['@operation'] === 'test_operation';
        })
      );

    $this->ociLogger->logOperation('test_operation');
  }

  /**
   * Tests logging errors.
   *
   * @covers ::logError
   */
  public function testLogError(): void {
    $exception = new \Exception('Test error');

    $this->mockChannel->expects($this->once())
      ->method('error');

    $this->ociLogger->logError('Test message', $exception);
  }

  /**
   * Tests logging security events.
   *
   * @covers ::logSecurityEvent
   */
  public function testLogSecurityEvent(): void {
    $this->mockChannel->expects($this->once())
      ->method('warning');

    $this->ociLogger->logSecurityEvent('Unauthorized access attempt');
  }

  /**
   * Tests sensitive data redaction.
   *
   * @covers ::logSdkCall
   */
  public function testSensitiveDataRedaction(): void {
    $this->mockChannel->expects($this->once())
      ->method('debug')
      ->with(
        $this->anything(),
        $this->callback(function ($context) {
          $params = json_decode($context['@params'], TRUE);
          return isset($params['private_key']) && $params['private_key'] === '[REDACTED]';
        })
      );

    $this->ociLogger->logSdkCall('testMethod', [
      'private_key' => 'secret-key-value',
      'bucket' => 'my-bucket',
    ]);
  }

}
