<?php

namespace Drupal\oembed_configuration;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Utility\Error;
use Drupal\media\OEmbed\ProviderRepositoryInterface;
use Drupal\media\OEmbed\ResourceFetcher;
use Drupal\media\OEmbed\ResourceFetcherInterface;
use GuzzleHttp\ClientInterface;

/**
 * Overrides the oEmbed resources.
 *
 * This class overrides the media.oembed.resource_fetcher service from the core
 * media module.
 *
 * @see https://www.drupal.org/project/drupal/issues/3042423
 */
class OEmbedResourceFetcher extends ResourceFetcher {

  /**
   * The inner service (ResourceFetcher service).
   *
   * @var \Drupal\media\OEmbed\ResourceFetcherInterface
   */
  protected ResourceFetcherInterface $innerService;

  /**
   * OEmbedResourceFetcher constructor.
   *
   * @param \Drupal\media\OEmbed\ResourceFetcherInterface $inner_service
   *   The resource fetcher service.
   * @param \GuzzleHttp\ClientInterface $httpClient
   *   The HTTP client.
   * @param \Drupal\media\OEmbed\ProviderRepositoryInterface $providers
   *   The oEmbed provider repository service.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cacheBackend
   *   The cache backend.
   * @param int $timeout
   *   The length of time to wait for the request before the request
   *   should time out.
   */
  public function __construct(ResourceFetcherInterface $inner_service, ClientInterface $httpClient, ProviderRepositoryInterface $providers, CacheBackendInterface $cacheBackend, int $timeout = 5) {
    $this->innerService = $inner_service;
    parent::__construct($httpClient, $providers, $cacheBackend, $timeout);
  }

  /**
   * {@inheritdoc}
   */
  protected function createResource(array $data, $url) {
    /**
* @var \Drupal\oembed_configuration\OEmbedConfigurationProviderPluginManager $manager
*/
    $manager = \Drupal::service('plugin.manager.oembed_configuration_provider');
    try {
      $instance = $manager->createInstance($data['provider_name']);
      if ($instance instanceof OEmbedConfigurationProviderInterface) {
        $instance->overrideValues($data);
      }
    }
    catch (\Exception $e) {
      $logger = \Drupal::logger('oembed_configuration');
      Error::logException($logger, $e);
    }
    return $this->innerService->createResource($data, $url);
  }

}
