<?php

namespace Drupal\oembed_configuration;

use Drupal\Component\Plugin\PluginBase;
use Drupal\Core\Cache\RefinableCacheableDependencyTrait;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Http\ClientFactory;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for oembed_configuration_provider plugins.
 */
abstract class OEmbedConfigurationProviderPluginBase extends PluginBase implements OEmbedConfigurationProviderInterface {

  use StringTranslationTrait;
  use RefinableCacheableDependencyTrait;

  /**
   * The configuration factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The key repository.
   *
   * @var \Drupal\key\KeyRepositoryInterface|null
   */
  protected $keyRepository;

  /**
   * The logger object.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  /**
   * Read-only config object.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $config;

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The HTTP client factory.
   *
   * @var \Drupal\Core\Http\ClientFactory
   */
  protected ClientFactory $httpClientFactory;

  /**
   * The state key/value store.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected StateInterface $state;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static($configuration, $plugin_id, $plugin_definition);
    $instance->configFactory = $container->get('config.factory');
    $instance->logger = $container->get('logger.factory')->get('oembed_configuration');
    $instance->moduleHandler = $container->get('module_handler');
    $instance->httpClientFactory = $container->get('http_client_factory');
    $instance->state = $container->get('state');
    $instance->keyRepository = NULL;

    if ($instance->moduleHandler->moduleExists('key')) {
      $instance->keyRepository = $container->get('key.repository');
    }

    $instance->config = $instance->configFactory->get(OEmbedConfigurationProviderInterface::OEMBED_CONFIGURATION_SETTINGS);
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function label(): string {
    // Cast the label to a string since it is a TranslatableMarkup object.
    return (string) $this->pluginDefinition['label'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form['override_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('URL to override'),
      '#description' => $this->t('This useful to use a more private (for example youtube-nocookie for YouTube), or to override the API version (useful for Meta products).'),
      '#default_value' => $this->config->get($this->pluginId)['override_url'] ?? $this->defaultConfiguration()['override_url'],
      '#weight' => 99,
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $config = $this->configFactory->getEditable(OEmbedConfigurationProviderInterface::OEMBED_CONFIGURATION_SETTINGS);
    $settings = [];
    foreach ($this->defaultConfiguration() as $key => $value) {
      $settings[$key] = $form_state->getValue($key) ?? $value;
    }
    $config->set($this->pluginId, $settings);
    $config->save();
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'override_url' => '',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function calculateDependencies() {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function applyQueryParams(array &$parsed_url): void {
    $parsed_url['query'] = array_merge(
          $parsed_url['query'],
          $this->config->get($this->getPluginId())['query'] ?? []
      );
  }

  /**
   * {@inheritdoc}
   */
  public function overrideUrl(array &$parsed_url): void {
    // Implemented in plugins.
  }

  /**
   * {@inheritdoc}
   */
  public function overrideValues(array &$data): void {
    // Implemented in plugins.
  }

}
