<?php

namespace Drupal\oembed_configuration\Plugin\OEmbedConfigurationProvider;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\key\KeyInterface;
use Drupal\oembed_configuration\Attribute\OEmbedConfigurationProvider;
use Drupal\oembed_configuration\OEmbedConfigurationProviderPluginBase;

/**
 * Plugin implementation of the oembed_configuration_provider for Instagram.
 *
 * @see https://developers.facebook.com/docs/instagram-platform/oembed/
 */
#[OEmbedConfigurationProvider(
    id: 'Instagram',
    label: new TranslatableMarkup('Instagram'),
    description: new TranslatableMarkup('Instagram OEmbed configuration.')
)]
class Instagram extends OEmbedConfigurationProviderPluginBase {

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form['query'] = [
      '#type' => 'details',
      '#title' => $this->t('Query parameters'),
    ];
    $form['query']['omitscript'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Omit script'),
      '#description' => $this->t("Don't include Instagram JavaScript in every post (it would need to be added globally)."),
      '#default_value' => $this->config->get($this->pluginId)['query']['omitscript'] ?? $this->defaultConfiguration()['query']['omitscript'],
    ];
    $form['auth'] = [
      '#type' => 'details',
      '#title' => $this->t('Authentication parameters'),
      '#description' => $this->t('Please note that Instagram requires authentication and the <a href="@url">key module</a> is highly recommended to manage it.', ['@url' => 'https://www.drupal.org/project/key']),
    ];
    $form['auth']['token_url'] = [
      '#type' => 'url',
      '#title' => $this->t('Token URL'),
      '#default_value' => $this->config->get($this->pluginId)['auth']['token_url'] ?? $this->defaultConfiguration()['auth']['token_url'],
    ];
    $form['auth']['grant_type'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Grant type'),
      '#default_value' => $this->config->get($this->pluginId)['auth']['grant_type'] ?? $this->defaultConfiguration()['auth']['grant_type'],
    ];
    $form['auth']['app_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('App ID'),
      '#default_value' => $this->config->get($this->pluginId)['auth']['app_id'] ?? $this->defaultConfiguration()['auth']['app_id'],
    ];
    $form['auth']['app_secret'] = [
      '#type' => 'textfield',
      '#title' => $this->t('App Secret'),
      '#default_value' => $this->config->get($this->pluginId)['auth']['app_secret'] ?? $this->defaultConfiguration()['auth']['app_secret'],
    ];
    if ($this->moduleHandler->moduleExists('key')) {
      $form['auth']['app_id']['#type'] = 'key_select';
      $form['auth']['app_id']['#key_filters'] = ['type' => 'authentication'];
      $form['auth']['app_id']['#empty_option'] = $this->t('- Please select -');
      $form['auth']['app_secret']['#type'] = 'key_select';
      $form['auth']['app_secret']['#key_filters'] = ['type' => 'authentication'];
      $form['auth']['app_secret']['#empty_option'] = $this->t('- Please select -');
    }
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return parent::defaultConfiguration() + [
      'query' => [
        'omitscript' => FALSE,
      ],
      'auth' => [
        'token_url' => 'https://graph.facebook.com/oauth/access_token',
        'grant_type' => 'client_credentials',
        'app_id' => '',
        'app_secret' => '',
      ],
      'override_url' => 'graph.facebook.com/v24.0/instagram_oembed',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function applyQueryParams(array &$parsed_url): void {
    parent::applyQueryParams($parsed_url);
    $access_token = $this->getToken();
    if (!empty($access_token)) {
      $parsed_url['query']['access_token'] = $access_token;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function overrideUrl(array &$parsed_url): void {
    $override_url = $this->config->get($this->getPluginId())['override_url'] ?? '';
    if (!empty($override_url)) {
      $parsed_url['path'] = preg_replace('/(:\/\/graph\.facebook\.com\/v\d+\.\d+\/instagram_oembed\/)/', '://' . $override_url . '/', $parsed_url['path']);
    }
  }

  /**
   * Calculates an access token from Meta.
   *
   * @return string
   *   The access token.
   *
   * @throws \Exception
   * @throws \GuzzleHttp\Exception\GuzzleException
   */
  protected function getToken(): string {
    // Token is stored in a state, so if it is current, we reuse it from there
    // to avoid unnecessary calls.
    $oauth_token = $this->state->get('oembed_configuration.meta_oauth_token');
    if (!empty($oauth_token['expires_on'])) {
      if ($oauth_token['expires_on'] > time()) {
        return $oauth_token['access_token'];
      }
    }
    $config = $this->config->get($this->getPluginId())['auth'] ?? [];
    if (empty($config)) {
      return '';
    }
    extract($config, \EXTR_SKIP);

    if ($this->moduleHandler->moduleExists('key')) {
      $key_app_id = $this->keyRepository->getKey($app_id);
      $key_app_secret = $this->keyRepository->getKey($app_secret);
      if (!$key_app_id instanceof KeyInterface || !$key_app_secret instanceof KeyInterface) {
        throw new \Exception('Client could not be initialized.');
      }
      $app_id = $key_app_id->getKeyValue();
      $app_secret = $key_app_secret->getKeyValue();
    }
    $endpoint = "$token_url?client_id=$app_id&client_secret=$app_secret&grant_type=$grant_type";

    $oauth_response = $this->httpClientFactory->fromOptions(['verify' => FALSE])
      ->request('GET', $endpoint, []);
    if ($oauth_response->getStatusCode() === 200) {
      $oauth_token = JSON::decode($oauth_response->getBody()->getContents());
      if (empty($oauth_token['expires_on'])) {
        $oauth_token['expires_on'] = strtotime('+1 week');
      }

        $this->state->set('oembed_configuration.meta_oauth_token', $oauth_token);
    }
    else {
      throw new \Exception('Client could not be initialized.');
    }

    return $oauth_token['access_token'] ?? '';
  }

}
