<?php

namespace Drupal\oembed_configuration\Plugin\OEmbedConfigurationProvider;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\oembed_configuration\Attribute\OEmbedConfigurationProvider;
use Drupal\oembed_configuration\OEmbedConfigurationProviderPluginBase;

/**
 * Plugin implementation of the oembed_configuration_provider for Vimeo.
 *
 * @see https://developer.vimeo.com/api/oembed/videos#before-you-begin
 */
#[OEmbedConfigurationProvider(
    id: 'Vimeo',
    label: new TranslatableMarkup('Vimeo'),
    description: new TranslatableMarkup('Vimeo OEmbed configuration.')
)]
class Vimeo extends OEmbedConfigurationProviderPluginBase {

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);
    $config_data = $this->config->get($this->pluginId);

    $form['query'] = [
      '#type' => 'details',
      '#title' => $this->t('Query parameters'),
    ];
    $form['query']['autoplay'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Autoplay'),
      '#description' => $this->t('Whether to start playback of the video automatically. This feature might not work on all devices.'),
      '#default_value' => $config_data['query']['autoplay'] ?? $this->defaultConfiguration()['query']['autoplay'],
    ];
    $form['query']['background'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Background'),
      '#description' => $this->t('Whether the player is in background mode, which hides the playback controls, enables autoplay, and loops the video.'),
      '#default_value' => $config_data['query']['background'] ?? $this->defaultConfiguration()['query']['background'],
    ];
    $form['query']['byline'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Byline'),
      '#description' => $this->t("Whether to display the video owner's name."),
      '#default_value' => $config_data['query']['byline'] ?? $this->defaultConfiguration()['query']['byline'],
    ];
    $form['query']['dnt'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Do not track'),
      '#description' => $this->t('Whether to prevent the player from tracking session data, including cookies. Keep in mind that setting this argument to true also blocks video stats.'),
      '#default_value' => $config_data['query']['dnt'] ?? $this->defaultConfiguration()['query']['dnt'],
    ];
    $form['query']['loop'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Loop'),
      '#description' => $this->t('Whether to restart the video automatically after reaching the end.'),
      '#default_value' => $config_data['query']['loop'] ?? $this->defaultConfiguration()['query']['loop'],
    ];
    $form['query']['responsive'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Responsive'),
      '#description' => $this->t('Whether to return a responsive embed code, or one that provides intelligent adjustments based on viewing conditions. We recommend this option for mobile-optimized sites.'),
      '#default_value' => $config_data['query']['responsive'] ?? $this->defaultConfiguration()['query']['responsive'],
    ];
    $form['query']['title'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Title'),
      '#description' => $this->t("Whether to display the video's title."),
      '#default_value' => $config_data['query']['title'] ?? $this->defaultConfiguration()['query']['title'],
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return parent::defaultConfiguration() + [
      'query' => [
        'autoplay' => FALSE,
        'background' => FALSE,
        'byline' => FALSE,
        'dnt' => TRUE,
        'loop' => FALSE,
        'responsive' => TRUE,
        'title' => TRUE,
      ],
    ];
  }

}
