<?php

namespace Drupal\oembed_configuration\Plugin\OEmbedConfigurationProvider;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\oembed_configuration\Attribute\OEmbedConfigurationProvider;
use Drupal\oembed_configuration\OEmbedConfigurationProviderPluginBase;

/**
 * Plugin implementation of the oembed_configuration_provider for YouTube.
 *
 * @see https://developers.google.com/youtube/player_parameters#Parameters
 */
#[OEmbedConfigurationProvider(
    id: 'YouTube',
    label: new TranslatableMarkup('YouTube'),
    description: new TranslatableMarkup('YouTube OEmbed configuration.')
)]
class YouTube extends OEmbedConfigurationProviderPluginBase {

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form['query'] = [
      '#type' => 'details',
      '#title' => $this->t('Query parameters'),
    ];
    $form['query']['autoplay'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Autoplay'),
      '#description' => $this->t('This parameter specifies whether the initial video will automatically start to play when the player loads. Supported values are 0 or 1. The default value is 0. If you enable Autoplay, playback will occur without any user interaction with the player; playback data collection and sharing will therefore occur upon page load.'),
      '#default_value' => $this->config->get($this->pluginId)['query']['autoplay'] ?? $this->defaultConfiguration()['query']['autoplay'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return parent::defaultConfiguration() + [
      'query' => [
        'autoplay' => FALSE,
      ],
      'override_url' => 'www.youtube-nocookie.com',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function overrideValues(array &$data): void {
    $override_url = $this->config->get($this->getPluginId())['override_url'] ?? '';
    if (!empty($override_url)) {
      // Replace the default YouTube domain with the no-cookie domain.
      $data['html'] = str_replace(['://youtube.com/', '://www.youtube.com/'], '://' . $override_url . '/', $data['html']);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function overrideUrl(array &$parsed_url): void {
    $override_url = $this->config->get($this->getPluginId())['override_url'] ?? '';
    if (!empty($override_url)) {
      // Replace the default YouTube domain with the no-cookie domain.
      $parsed_url['path'] = str_replace(['://youtube.com/', '://www.youtube.com/'], '://' . $override_url . '/', $parsed_url['path']);
    }
  }

}
