<?php

namespace Drupal\oembed_field\Plugin\Field\FieldFormatter;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\media\OEmbed\ResourceFetcherInterface;
use Drupal\media\OEmbed\UrlResolverInterface;

/**
 * Plugin implementation of the 'oembed_thumbnail' formatter.
 *
 * @FieldFormatter(
 *   id = "oembed_field_thumbnail",
 *   label = @Translation("oEmbed thumbnail"),
 *   field_types = {
 *     "oembed_url_field"
 *   }
 * )
 */
class OembedFieldThumbnailFormatter extends FormatterBase {

  /**
   * The logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The oEmbed resource fetcher.
   *
   * @var \Drupal\media\OEmbed\ResourceFetcherInterface
   */
  protected $resourceFetcher;

  /**
   * The oEmbed URL resolver.
   *
   * @var \Drupal\media\OEmbed\UrlResolverInterface
   */
  protected $urlResolver;

  /**
   * Constructs an OembedThumbnailFormatter instance.
   */
  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    $label,
    $view_mode,
    array $third_party_settings,
    LoggerChannelFactoryInterface $logger_factory,
    ResourceFetcherInterface $resource_fetcher,
    UrlResolverInterface $url_resolver,
  ) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);
    $this->logger = $logger_factory->get('oembed_field');
    $this->resourceFetcher = $resource_fetcher;
    $this->urlResolver = $url_resolver;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('logger.factory'),
      $container->get('media.oembed.resource_fetcher'),
      $container->get('media.oembed.url_resolver')
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'image_style' => '',
      'image_link' => 'content',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = parent::settingsForm($form, $form_state);

    $image_styles = image_style_options(FALSE);
    $elements['image_style'] = [
      '#title' => $this->t('Image style'),
      '#type' => 'select',
      '#default_value' => $this->getSetting('image_style'),
      '#empty_option' => $this->t('None (original image)'),
      '#options' => $image_styles,
      '#description' => $this->t('Select an image style to apply to the thumbnail.'),
    ];

    $link_types = [
      'nothing' => $this->t('Nothing'),
      'content' => $this->t('Content'),
      'provider' => $this->t('Provider URL'),
      'file' => $this->t('File'),
    ];
    $elements['image_link'] = [
      '#title' => $this->t('Link image to'),
      '#type' => 'select',
      '#default_value' => $this->getSetting('image_link'),
      '#options' => $link_types,
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];

    $image_styles = image_style_options(FALSE);
    $image_style_setting = $this->getSetting('image_style');
    if (isset($image_styles[$image_style_setting])) {
      $summary[] = $this->t('Image style: @style', [
        '@style' => $image_styles[$image_style_setting],
      ]);
    }
    else {
      $summary[] = $this->t('Original image');
    }

    $link_setting = $this->getSetting('image_link');
    $link_types = [
      'nothing' => $this->t('Not linked'),
      'content' => $this->t('Linked to content'),
      'provider' => $this->t('Linked to provider'),
      'file' => $this->t('Linked to file'),
    ];
    $summary[] = $link_types[$link_setting];

    return $summary;
  }

  /**
   * {@inheritdoc}
   */

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];
    $thumbnail_manager = \Drupal::service('oembed_field.thumbnail_manager');

    // Add cache tags for the entity and field.
    $entity = $items->getEntity();
    $cache_metadata = new CacheableMetadata();
    $cache_metadata->addCacheTags($entity->getCacheTags());
    $cache_metadata->addCacheTags(['oembed_thumbnails_list']);

    foreach ($items as $delta => $item) {
      if (empty($item->value) || empty($item->url_hash)) {
        continue;
      }

      // Get thumbnail file from manager (it will download if needed).
      $file = $item->getThumbnailFile();

      if (!$file) {
        continue;
      }

      $thumbnail_uri = $file->getFileUri();
      $alt = '';

      // Apply image style if configured.
      if ($this->getSetting('image_style')) {
        $image = [
          '#theme' => 'image_style',
          '#style_name' => $this->getSetting('image_style'),
          '#uri' => $thumbnail_uri,
          '#alt' => $alt,
        ];
      }
      else {
        $image = [
          '#theme' => 'image',
          '#uri' => $thumbnail_uri,
          '#alt' => $alt,
        ];
      }

      // Add link if configured.
      $link_setting = $this->getSetting('image_link');
      switch ($link_setting) {
        case 'provider':
          $elements[$delta] = [
            '#type' => 'link',
            '#title' => $image,
            '#url' => Url::fromUri($item->value),
          ];
          break;

        case 'content':
          $elements[$delta] = [
            '#type' => 'link',
            '#title' => $image,
            '#url' => $item->getEntity()->toUrl(),
          ];
          break;

        case 'file':
          $url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
          $elements[$delta] = [
            '#type' => 'link',
            '#title' => $image,
            '#url' => Url::fromUri($url),
          ];
          break;

        default:
          $elements[$delta] = $image;
      }

      // Apply cache metadata to each element.
      $cache_metadata->applyTo($elements[$delta]);
    }

    return $elements;
  }

}
