<?php

namespace Drupal\oembed_field;

use Drupal\Core\Config\ConfigFactory;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandler;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\media\OEmbed\ProviderRepositoryInterface;
use Drupal\media\OEmbed\ResourceFetcherInterface;
use Drupal\media\OEmbed\UrlResolverInterface;

/**
 * Class AffiliateManager service.
 */
class OembedFieldManager {

  use StringTranslationTrait;

  /**
   * The EntityTypeManagerInterface.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The request stack.
   *
   * @var \Drupal\Core\Extension\ModuleHandler
   */
  protected $moduleHandler;

  /**
   * The affiliate config settings.
   *
   * @var \Drupal\Core\Config\Config|\Drupal\Core\Config\ImmutableConfig
   */
  protected $config;

  /**
   * @var \Drupal\media\OEmbed\ResourceFetcherInterface
   */
  protected $resourceFetcher;

  /**
   * @var \Drupal\media\OEmbed\UrlResolverInterface
   */
  protected $urlResolver;

  /**
   * @var \Drupal\media\OEmbed\ProviderRepositoryInterface
   */
  protected $providerRepository;

  /**
   * Constructs an AffiliateManager service.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    ModuleHandler $module_handler,
    ConfigFactory $config_factory,
    ResourceFetcherInterface $resource_fetcher,
    UrlResolverInterface $url_resolver,
    ProviderRepositoryInterface $provider_repository,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->moduleHandler = $module_handler;
    $this->config = $config_factory->get('affiliated.settings');
    $this->resourceFetcher = $resource_fetcher;
    $this->urlResolver = $url_resolver;
    $this->providerRepository = $provider_repository;
  }

  /**
   * Check if URL is a valid oEmbed URL.
   */
  public function isValidOembedUrl(string $url, array $allowed_buckets): bool {
    // If no buckets configured, check against ALL providers.
    if (empty($allowed_buckets)) {
      $providers = $this->providerRepository->getAll();
    }
    else {
      // Get only providers from allowed buckets.
      $providers = $this->getProvidersFromBuckets($allowed_buckets);
    }

    // Check if URL matches any provider.
    foreach ($providers as $provider_info) {
      foreach ($provider_info->getEndpoints() as $endpoint) {
        if ($endpoint->matchUrl($url)) {
          return TRUE;
        }
      }
    }

    return FALSE;
  }

  /**
   * Returns an array of all providers in $buckets.
   */
  public function getProvidersFromBuckets(array $buckets): array {
    $enabled_providers = $this->getProviderNamesFromBuckets($buckets);
    foreach ($enabled_providers as $provider) {
      $providers[] = $this->providerRepository->get($provider);
    }
    return $providers;
  }

  /**
   *
   */
  public function getProviderNamesFromBuckets(array $buckets): array {
    $enabled_providers = [];
    $buckets = $this->entityTypeManager->getStorage('oembed_provider_bucket')
      ->loadMultiple($buckets);
    foreach ($buckets as $bucket) {
      $enabled_providers = array_merge($enabled_providers, $bucket->get('providers'));
    }
    return array_unique($enabled_providers);
  }

}
