<?php

namespace Drupal\oembed_field;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Component\Render\PlainTextOutput;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\File\Exception\FileException;
use Drupal\Core\File\FileExists;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Lock\LockBackendInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Utility\Token;
use Drupal\media\OEmbed\ResourceFetcherInterface;
use Drupal\media\OEmbed\UrlResolverInterface;
use Psr\Http\Client\ClientInterface;
use Psr\Http\Client\ClientExceptionInterface;
use Psr\Http\Message\ResponseInterface;
use Symfony\Component\Mime\MimeTypes;

/**
 * Manages centralized thumbnail storage for oEmbed fields.
 */
class OembedThumbnailManager {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The HTTP client.
   *
   * @var \Psr\Http\Client\ClientInterface
   */
  protected $httpClient;

  /**
   * The oEmbed resource fetcher.
   *
   * @var \Drupal\media\OEmbed\ResourceFetcherInterface
   */
  protected $resourceFetcher;

  /**
   * The oEmbed URL resolver.
   *
   * @var \Drupal\media\OEmbed\UrlResolverInterface
   */
  protected $urlResolver;

  /**
   * The logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The lock backend.
   *
   * @var \Drupal\Core\Lock\LockBackendInterface
   */
  protected $lock;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The token service.
   *
   * @var \Drupal\Core\Utility\Token
   */
  protected $token;

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   * @param \Psr\Http\Client\ClientInterface $http_client
   *   The HTTP client.
   * @param \Drupal\media\OEmbed\ResourceFetcherInterface $resource_fetcher
   *   The oEmbed resource fetcher.
   * @param \Drupal\media\OEmbed\UrlResolverInterface $url_resolver
   *   The oEmbed URL resolver.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Lock\LockBackendInterface $lock
   *   The lock backend.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Drupal\Core\Utility\Token $token
   *   The token service.
   */
  public function __construct(
    Connection $database,
    EntityTypeManagerInterface $entity_type_manager,
    FileSystemInterface $file_system,
    ClientInterface $http_client,
    ResourceFetcherInterface $resource_fetcher,
    UrlResolverInterface $url_resolver,
    LoggerChannelFactoryInterface $logger_factory,
    LockBackendInterface $lock,
    TimeInterface $time,
    ModuleHandlerInterface $module_handler,
    Token $token,
  ) {
    $this->database = $database;
    $this->entityTypeManager = $entity_type_manager;
    $this->fileSystem = $file_system;
    $this->httpClient = $http_client;
    $this->resourceFetcher = $resource_fetcher;
    $this->urlResolver = $url_resolver;
    $this->logger = $logger_factory->get('oembed_field');
    $this->lock = $lock;
    $this->time = $time;
    $this->moduleHandler = $module_handler;
    $this->token = $token;
  }

  /**
   * Get thumbnail file for a field item.
   *
   * @param \Drupal\Core\Field\FieldItemInterface $field_item
   *   The field item.
   *
   * @return \Drupal\file\Entity\File|null
   *   The file entity or NULL if not available.
   */
  public function getThumbnailFileForField($field_item) {
    $url_hash = $field_item->get('url_hash')->getValue();
    $url = $field_item->get('value')->getValue();

    if (!$url_hash || !$url) {
      return NULL;
    }

    // First check if we already have the file.
    $file = $this->getThumbnailFileFromHash($url_hash);
    if ($file) {
      return $file;
    }

    // No file yet, need to fetch thumbnail URL from oEmbed.
    try {
      $resource_url = $this->urlResolver->getResourceUrl($url, 1000, 1000);
      $resource = $this->resourceFetcher->fetchResource($resource_url);

      if ($resource->getThumbnailUrl()) {
        $thumbnail_url = $resource->getThumbnailUrl()->toString();

        // Get settings from the field item.
        $settings = $field_item->getThumbnailSettings();
        $entity = $field_item->getEntity();

        return $this->downloadAndStoreThumbnail($url_hash, $thumbnail_url, $settings, $entity);
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Could not fetch thumbnail for @url: @error', [
        '@url' => $url,
        '@error' => $e->getMessage(),
      ]);
    }

    return NULL;
  }

  /**
   * Get thumbnail file for a given URL hash.
   *
   * @param string $url_hash
   *   The hash of the oEmbed URL.
   *
   * @return \Drupal\file\Entity\File|null
   *   The file entity or NULL if not available.
   */
  public function getThumbnailFileFromHash($url_hash) {
    // Check if we have it in the database.
    $record = $this->database->select('oembed_field_thumbnails', 't')
      ->fields('t', ['file_id'])
      ->condition('url_hash', $url_hash)
      ->execute()
      ->fetch();

    if ($record && $record->file_id) {
      /** @var \Drupal\file\Entity\File $file */
      $file = $this->entityTypeManager->getStorage('file')->load($record->file_id);
      if ($file) {
        return $file;
      }
    }
    return NULL;
  }

  /**
   * Download and store a thumbnail.
   *
   * @param string $url_hash
   *   The hash of the oEmbed URL.
   * @param string $thumbnail_url
   *   The thumbnail URL to download.
   * @param array $settings
   *   Field settings for directory and URI scheme.
   *
   * @return \Drupal\file\Entity\File|null
   *   The created file entity or NULL on failure.
   */
  protected function downloadAndStoreThumbnail($url_hash, $thumbnail_url, array $settings, $entity = NULL) {
    // Use a lock to prevent concurrent downloads of the same thumbnail.
    $lock_name = 'oembed_thumbnail_download:' . $url_hash;

    // Try to acquire lock (wait up to 30 seconds).
    if ($this->lock->acquire($lock_name, 30)) {
      try {
        // Double-check after acquiring lock - another process might have
        // downloaded it.
        $record = $this->database->select('oembed_field_thumbnails', 't')
          ->fields('t', ['file_id'])
          ->condition('url_hash', $url_hash)
          ->execute()
          ->fetch();

        if ($record && $record->file_id) {
          /** @var \Drupal\file\Entity\File $file */
          $file = $this->entityTypeManager->getStorage('file')->load($record->file_id);
          if ($file) {
            return $file;
          }
        }

        // Insert a placeholder record to indicate download is in progress.
        // This prevents other processes from trying to download simultaneously.
        $this->database->merge('oembed_field_thumbnails')
          ->key(['url_hash' => $url_hash])
          ->fields([
            // NULL indicates "downloading".
            'file_id' => NULL,
            'created' => $this->time->getRequestTime(),
          ])
          ->execute();

        // Get directory and URI scheme from settings or use defaults.
        $uri_scheme = $settings['uri_scheme'] ?? 'public';
        $file_directory = $settings['file_directory'] ?? 'oembed_thumbnails';

        // Try to get the created date for token replacement from the entity
        // the field is attached to. Fallback to current time.
        if ($entity && $entity->hasField('created')) {
          $timestamp = $entity->get('created')->value;
        }
        $token_data = ['date' => $timestamp ?? $this->time->getCurrentTime()];

        // Replace tokens in directory path.
        if ($this->moduleHandler->moduleExists('token')) {
          $file_directory = $this->token->replace($file_directory, $token_data);
        }
        $file_directory = PlainTextOutput::renderFromHtml($file_directory);

        // Create the full directory path.
        $directory = $uri_scheme . '://' . $file_directory;

        if (!$this->fileSystem->prepareDirectory($directory, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS)) {
          $this->logger->warning('Could not prepare thumbnail directory @dir.', [
            '@dir' => $directory,
          ]);
          return NULL;
        }

        // Download the thumbnail.
        $response = $this->httpClient->request('GET', $thumbnail_url, [
          'timeout' => 30,
        ]);
        if ($response->getStatusCode() !== 200) {
          return NULL;
        }

        // Determine file extension.
        $extension = $this->getThumbnailFileExtension($thumbnail_url, $response);
        $filename = $url_hash . '.' . $extension;
        $uri = $directory . '/' . $filename;

        // Save the file.
        $this->fileSystem->saveData((string) $response->getBody(), $uri, FileExists::Replace);

        // Create file entity.
        $file = $this->entityTypeManager->getStorage('file')->create([
          'uri' => $uri,
          'status' => 1,
        ]);
        $file->save();

        // Store in our tracking table.
        $this->database->merge('oembed_field_thumbnails')
          ->key(['url_hash' => $url_hash])
          ->fields([
            'file_id' => $file->id(),
            'created' => $this->time->getRequestTime(),
          ])
          ->execute();

        return $file;
      }
      catch (ClientExceptionInterface $e) {
        $this->logger->warning('Failed to download thumbnail from @url: @error', [
          '@url' => $thumbnail_url,
          '@error' => $e->getMessage(),
        ]);
      }
      catch (FileException $e) {
        $this->logger->warning('Could not save thumbnail file: @error', [
          '@error' => $e->getMessage(),
        ]);
      }
      finally {
        $this->lock->release($lock_name);
      }
    }
    else {
      // Couldn't get lock, another process is downloading.
      // Wait a bit and check if it's done.
      sleep(2);

      $record = $this->database->select('oembed_field_thumbnails', 't')
        ->fields('t', ['file_id'])
        ->condition('url_hash', $url_hash)
        ->execute()
        ->fetch();

      if ($record && $record->file_id) {
        $file = $this->entityTypeManager->getStorage('file')->load($record->file_id);
        if ($file) {
          return $file;
        }
      }
    }

    return NULL;
  }

  /**
   * Determine file extension from URL or response.
   *
   * @param string $thumbnail_url
   *   The thumbnail URL.
   * @param \Psr\Http\Message\ResponseInterface $response
   *   The HTTP response.
   *
   * @return string
   *   The file extension.
   */
  protected function getThumbnailFileExtension(string $thumbnail_url, ResponseInterface $response): string {
    // Try to get from URL.
    $path = parse_url($thumbnail_url, PHP_URL_PATH);
    if ($path) {
      $extension = strtolower(pathinfo($path, PATHINFO_EXTENSION));
      if ($extension) {
        return $extension;
      }
    }

    // Try to get from Content-Type header.
    $content_type = $response->getHeader('Content-Type');
    if (!empty($content_type)) {
      $extensions = MimeTypes::getDefault()->getExtensions(reset($content_type));
      if ($extensions) {
        return reset($extensions);
      }
    }

    // Default to jpg.
    return 'jpg';
  }

}
