<?php

namespace Drupal\oembed_field\Plugin\Field\FieldFormatter;

use Drupal\Component\Utility\Html;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\media\IFrameMarkup;
use Drupal\media\IFrameUrlHelper;
use Drupal\media\OEmbed\Resource;
use Drupal\media\OEmbed\ResourceException;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\media\OEmbed\ResourceFetcherInterface;
use Drupal\media\OEmbed\UrlResolverInterface;
use Drupal\Core\Render\RendererInterface;

/**
 * Plugin implementation of the 'oembed_default' formatter.
 *
 * @FieldFormatter(
 *   id = "oembed_field_default",
 *   label = @Translation("oEmbed content"),
 *   field_types = {
 *     "oembed_url_field"
 *   }
 * )
 */
class OembedFieldDefaultFormatter extends FormatterBase {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The oEmbed resource fetcher.
   *
   * @var \Drupal\media\OEmbed\ResourceFetcherInterface
   */
  protected $resourceFetcher;

  /**
   * The oEmbed URL resolver.
   *
   * @var \Drupal\media\OEmbed\UrlResolverInterface
   */
  protected $urlResolver;

  /**
   * The iFrame URL helper.
   *
   * @var \Drupal\media\IFrameUrlHelper
   */
  protected $iFrameUrlHelper;

  /**
   * The renderer.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * Constructs an OembedDefaultFormatter instance.
   */
  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    $label,
    $view_mode,
    array $third_party_settings,
    ConfigFactoryInterface $config_factory,
    LoggerChannelFactoryInterface $logger_factory,
    ResourceFetcherInterface $resource_fetcher,
    UrlResolverInterface $url_resolver,
    IFrameUrlHelper $iframe_url_helper,
    RendererInterface $renderer,
  ) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);
    $this->configFactory = $config_factory;
    $this->logger = $logger_factory->get('oembed_field');
    $this->resourceFetcher = $resource_fetcher;
    $this->urlResolver = $url_resolver;
    $this->iFrameUrlHelper = $iframe_url_helper;
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('config.factory'),
      $container->get('logger.factory'),
      $container->get('media.oembed.resource_fetcher'),
      $container->get('media.oembed.url_resolver'),
      $container->get('media.oembed.iframe_url_helper'),
      $container->get('renderer')
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'max_width' => 0,
      'max_height' => 0,
      'use_iframe' => TRUE,
      'enablejsapi' => FALSE,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = parent::settingsForm($form, $form_state);

    $elements['max_width'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum width'),
      '#default_value' => $this->getSetting('max_width'),
      '#min' => 0,
      '#description' => $this->t('Use zero to use original values. Both width and height should be set to properly constrain oEmbed resources.'),
    ];

    $elements['max_height'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum height'),
      '#default_value' => $this->getSetting('max_height'),
      '#min' => 0,
      '#description' => $this->t('Use zero to use original values. Both width and height should be set to properly constrain oEmbed resources.'),
    ];

    $elements['use_iframe'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use secure iframe'),
      '#default_value' => $this->getSetting('use_iframe'),
      '#description' => $this->t('Render embedded content in a secure iframe (recommended for security). If this is unchecked, the content will be rendered from the oembed returned html.'),
    ];

    $elements['enablejsapi'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable YouTube JS API'),
      '#default_value' => $this->getSetting('enablejsapi'),
      '#description' => $this->t('Enables javascript API for YouTube videos. This is required for the video player to be controlled via JavaScript.'),
      '#states' => [
        'visible' => [
          ':input[name*="[use_iframe]"]' => ['checked' => FALSE],
        ],
      ],
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];

    if ($max_width = $this->getSetting('max_width')) {
      $summary[] = $this->t('Max width: @width', ['@width' => $max_width]);
    }

    if ($max_height = $this->getSetting('max_height')) {
      $summary[] = $this->t('Max height: @height', ['@height' => $max_height]);
    }

    $summary[] = $this->getSetting('use_iframe')
      ? $this->t('Using secure iframe')
      : $this->t('Direct embed');

    if (!$this->getSetting('use_iframe') && $this->getSetting('enablejsapi')) {
      $summary[] = $this->t('Enable YouTube JS API');
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];
    $max_width = $this->getSetting('max_width');
    $max_height = $this->getSetting('max_height');

    $entity = $items->getEntity();
    $cache_metadata = new CacheableMetadata();
    $cache_metadata->addCacheTags($entity->getCacheTags());
    $cache_metadata->addCacheTags($items->getFieldDefinition()->getCacheTags());

    foreach ($items as $delta => $item) {
      $value = $item->value ?? NULL;
      if (empty($value)) {
        continue;
      }

      $resource = NULL;
      try {
        $resource_url = $this->urlResolver->getResourceUrl($value, $max_width, $max_height);
        $resource = $this->resourceFetcher->fetchResource($resource_url);
      }
      catch (ResourceException $e) {
        $this->logger->error('Could not retrieve oEmbed resource: @error', [
          '@error' => $e->getMessage(),
        ]);
      }

      if ($resource) {
        $elements[$delta] = $this->buildElementFromResource($resource, $value, $max_width, $max_height);
      }
      else {
        // Fallback to a simple link.
        $elements[$delta] = [
          '#type' => 'link',
          '#title' => $item->value,
          '#url' => Url::fromUri($value),
        ];
      }

      $cache_metadata->applyTo($elements[$delta]);
    }

    return $elements;
  }

  /**
   * Build render element from resource.
   */
  protected function buildElementFromResource($resource, $url, $max_width, $max_height) {
    $type = $resource->getType();

    switch ($type) {
      case Resource::TYPE_PHOTO:
        return [
          '#theme' => 'image',
          '#uri' => $resource->getUrl()->toString(),
          '#width' => $max_width ?: $resource->getWidth(),
          '#height' => $max_height ?: $resource->getHeight(),
          '#alt' => $resource->getTitle(),
        ];

      case Resource::TYPE_VIDEO:
      case Resource::TYPE_RICH:
        if ($this->getSetting('use_iframe')) {
          $data = [
            'width' => $resource->getWidth(),
            'height' => $resource->getHeight(),
            'title' => $resource->getTitle(),
          ];
          return $this->buildIframe($url, $max_width, $max_height, $data);
        }
        else {
          // Modify the oEmbed HTML for YouTube JS API if needed.
          $html = $resource->getHtml();
          $dom = Html::load($html);
          $iframe = $dom->getElementsByTagName('iframe')->item(0);
          $provider_name = $resource->getProvider()->getName();

          if ($iframe) {
            if ($provider_name === 'YouTube' && $this->getSetting('enablejsapi')) {
              $src = $iframe->getAttribute('src');
              $parts = parse_url($src);
              if (isset($parts['query'])) {
                parse_str($parts['query'], $query);
                $query['enablejsapi'] = 1;
                $query['origin'] = \Drupal::request()->getSchemeAndHttpHost();
                $new_src = Url::fromUri($src)->setOption('query', $query);
                $iframe->setAttribute('src', $new_src->toString());
              }
            }
            if (!$iframe->getAttribute('id')) {
              $id = Html::getUniqueId($provider_name . '_' . hash('crc32b', $url));
              $iframe->setAttribute('id', $id);
            }
            $html = Html::serialize($dom);
          }

          return [
            '#type' => 'inline_template',
            '#template' => '<div class="oembed-field-embed provider-{{ provider }}">{{ html|raw }}</div>',
            '#context' => [
              'html' => IFrameMarkup::create($html),
              'provider' => Html::cleanCssIdentifier($provider_name),
            ],
          ];
        }

      case Resource::TYPE_LINK:
      default:
        return [
          '#type' => 'link',
          '#title' => $resource->getTitle(),
          '#url' => Url::fromUri($url),
        ];
    }
  }

  /**
   * Build iframe element.
   */
  protected function buildIframe($url, $max_width, $max_height, $data) {
    $url_object = Url::fromRoute('media.oembed_iframe', [], [
      'query' => [
        'url' => $url,
        'max_width' => $max_width,
        'max_height' => $max_height,
        'hash' => $this->iFrameUrlHelper->getHash($url, $max_width, $max_height),
      ],
    ]);

    $domain = $this->configFactory->get('media.settings')->get('iframe_domain');
    if ($domain) {
      $url_object->setOption('base_url', $domain);
    }

    return [
      '#type' => 'html_tag',
      '#tag' => 'iframe',
      '#attributes' => [
        'src' => $url_object->toString(),
        'frameborder' => 0,
        'scrolling' => FALSE,
        'allowtransparency' => TRUE,
        'width' => $data['width'] ?? $max_width,
        'height' => $data['height'] ?? $max_height,
        'class' => ['oembed-field-iframe'],
        'title' => $data['title'] ?? $this->t('Embedded content'),
      ],
      '#attached' => [
        'library' => [
          'media/oembed.formatter',
        ],
      ],
    ];
  }

}
